//===========================================================================
// @(#) $DwmPath: dwm/DwmDns/tags/DwmDns-0.2.5/classes/include/DwmDnsRRDataNSEC3.hh 10129 $
// @(#) $Id: DwmDnsRRDataNSEC3.hh 10129 2018-01-27 04:20:26Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2018
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmDnsRRDataNSEC3.hh
//!  \brief Dwm::Dns::RRDataNSEC3
//---------------------------------------------------------------------------

#ifndef _DWMDNSRRDATANSEC3_HH_
#define _DWMDNSRRDATANSEC3_HH_

#include <cstdint>
#include <iostream>
#include <string>

#include "DwmDnsNSECBitmap.hh"
#include "DwmDnsRRData.hh"

namespace Dwm {

  namespace Dns {

    //------------------------------------------------------------------------
    //!  @ingroup rrdatagroup
    //!  Encapsulates NSEC3 resource record data.  See RFC5155 for details.
    //------------------------------------------------------------------------
    class RRDataNSEC3
      : public RRData
    {
    public:
      static const uint16_t  k_rrtype = 50;
      
      //----------------------------------------------------------------------
      //!  Construct with empty salt, empty hashed owner name and empty
      //!  bitmaps.
      //----------------------------------------------------------------------
      RRDataNSEC3();

      //----------------------------------------------------------------------
      //!  Construct with the given @c hashAlgorithm, @c flags, @c iterations,
      //!  @c salt, @c hashedOwnerName and @c bitmaps.
      //----------------------------------------------------------------------
      RRDataNSEC3(uint8_t hashAlgorithm, uint8_t flags, uint16_t iterations,
                  const std::string & salt,
                  const std::string & hashedOwnerName,
                  const std::vector<NSECBitmap> & bitmaps);

      //----------------------------------------------------------------------
      //!  Equal-to operator, mostly for unit testing.
      //----------------------------------------------------------------------
      bool operator == (const RRDataNSEC3 & nsec3) const;

      //----------------------------------------------------------------------
      //!  Returns the contained hash algorithm.
      //----------------------------------------------------------------------
      uint8_t HashAlgorithm() const;

      //----------------------------------------------------------------------
      //!  Sets and returns the contained hash algorithm.
      //----------------------------------------------------------------------
      uint8_t HashAlgorithm(uint8_t hashAlgorithm);

      //----------------------------------------------------------------------
      //!  Returns the contained flags.
      //----------------------------------------------------------------------
      uint8_t Flags() const;

      //----------------------------------------------------------------------
      //!  Sets and returns the contained flags.
      //----------------------------------------------------------------------
      uint8_t Flags(uint8_t flags);

      //----------------------------------------------------------------------
      //!  Returns the contained iterations.
      //----------------------------------------------------------------------
      uint16_t Iterations() const;

      //----------------------------------------------------------------------
      //!  Sets and returns the contained iterations.
      //----------------------------------------------------------------------
      uint16_t Iterations(uint16_t iterations);

      //----------------------------------------------------------------------
      //!  Returns the contained salt.
      //----------------------------------------------------------------------
      const std::string & Salt() const;

      //----------------------------------------------------------------------
      //!  Sets and returns the contained salt.
      //----------------------------------------------------------------------
      const std::string & Salt(const std::string & salt);

      //----------------------------------------------------------------------
      //!  Returns the contained hashed owner name.
      //----------------------------------------------------------------------
      const std::string & HashedOwnerName() const;

      //----------------------------------------------------------------------
      //!  Sets and returns the contained hashed owner name.
      //----------------------------------------------------------------------
      const std::string & HashedOwnerName(const std::string & hashedOwnerName);

      //----------------------------------------------------------------------
      //!  Returns the contained bitmaps.
      //----------------------------------------------------------------------
      const std::vector<NSECBitmap> & Bitmaps() const;

      //----------------------------------------------------------------------
      //!  Sets and returns the contained bitmaps.
      //----------------------------------------------------------------------
      const std::vector<NSECBitmap> &
      Bitmaps(const std::vector<NSECBitmap> & bitmaps);
      
      //----------------------------------------------------------------------
      //!  Encodes the NSEC3 resource record data into buffer @ pkt of length
      //!  @c pktlen, starting at @c ptr.  Returns the address immediately
      //!  following the encoded NSEC3 resource record data in @c pkt on
      //!  success.  Throws std::out_of_range if @c pkt is too short to
      //!  contain the encoded NSEC3 resource record data.
      //----------------------------------------------------------------------
      uint8_t *Encode(uint8_t *pkt, uint8_t *ptr, uint16_t pktlen,
                      LabelPositions & lps) const;

      //----------------------------------------------------------------------
      //!  Decodes the NSEC3 resource record data from buffer @c pkt of length
      //!  @c pktlen, starting at @c ptr.  Returns the address immediately
      //!  following the encoded  NSEC3 resource record data in @c pkt on
      //!  success.  Throws std::out_of_range if @c pkt was too short to
      //!  contain an encoded NSEC3 resource record data.
      //----------------------------------------------------------------------
      const uint8_t *Decode(const uint8_t *pkt, const uint8_t *ptr,
                            uint16_t pktlen, uint16_t rdlen);

      //----------------------------------------------------------------------
      //!  Print the NSEC3 resource record data to an ostream in human
      //!  readable form.
      //----------------------------------------------------------------------
      friend std::ostream & operator << (std::ostream & os,
                                         const RRDataNSEC3 & nsec3);
      
    private:
      uint8_t                  _hashAlgorithm;
      uint8_t                  _flags;
      uint16_t                 _iterations;
      std::string              _salt;
      std::string              _hashedOwnerName;
      std::vector<NSECBitmap>  _bitmaps;
    };
    
  }  // namespace Dns

}  // namespace Dwm

#endif  // _DWMDNSRRDATANSEC3_HH_
