//===========================================================================
// @(#) $DwmPath: dwm/DwmDns/tags/DwmDns-0.2.5/classes/include/DwmDnsRRDataRRSIG.hh 10137 $
// @(#) $Id: DwmDnsRRDataRRSIG.hh 10137 2018-01-28 07:25:17Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2018
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmDnsRRDataRRSIG.hh
//!  \brief Dwm::Dns::RRDataRRSIG class definition
//---------------------------------------------------------------------------

#ifndef _DWMDNSRRDATARRSIG_HH_
#define _DWMDNSRRDATARRSIG_HH_

#include <cstdint>
#include <iostream>
#include <string>

#include "DwmDnsRRData.hh"

namespace Dwm {

  namespace Dns {

    //------------------------------------------------------------------------
    //!  @ingroup rrdatagroup
    //!  Encapsulates RRSIG resource record data.  See RFC4034 for details.
    //------------------------------------------------------------------------
    class RRDataRRSIG
      : public RRData
    {
    public:
      static const uint16_t  k_rrtype = 46;

      //!@{
      //!  Known algorithm (from RFC4034 Appendix A.1)
      static constexpr uint8_t  k_algoRSAMD5     = 1;
      static constexpr uint8_t  k_algoDH         = 2;
      static constexpr uint8_t  k_algoDSA        = 3;
      static constexpr uint8_t  k_algoECC        = 4;
      static constexpr uint8_t  k_algoRSASHA1    = 5;
      static constexpr uint8_t  k_algoINDIRECT   = 252;
      static constexpr uint8_t  k_algoPRIVATEDNS = 253;
      static constexpr uint8_t  k_algoPRIVATEOID = 254;
      //!@}
      
      //----------------------------------------------------------------------
      //!  Construct with empty signer's name and signature.
      //----------------------------------------------------------------------
      RRDataRRSIG();
      
      //----------------------------------------------------------------------
      //!  Construct with the given @c typeCovered, @c algorithm, @c labels,
      //!  @c originalTTL, @c signatureExpiration, @c signatureInception,
      //!  @c keyTag, @c signersName and @c signature.
      //----------------------------------------------------------------------
      RRDataRRSIG(uint16_t typeCovered, uint8_t algorithm, uint8_t labels,
                  uint32_t originalTTL, uint32_t signatureExpiration,
                  uint32_t signatureInception, uint16_t keyTag,
                  const std::string & signersName,
                  const std::string & signature);
      
      //----------------------------------------------------------------------
      //!  Equal-to operator, mostly for unit testing.
      //----------------------------------------------------------------------
      bool operator == (const RRDataRRSIG & rrsig) const;
      
      //----------------------------------------------------------------------
      //!  Returns the contained type covered.
      //----------------------------------------------------------------------
      uint16_t TypeCovered() const;
      
      //----------------------------------------------------------------------
      //!  Sets and returns the contained type covered.
      //----------------------------------------------------------------------
      uint16_t TypeCovered(uint16_t typeCovered);
      
      //----------------------------------------------------------------------
      //!  Returns the contained algorithm.
      //----------------------------------------------------------------------
      uint8_t Algorithm() const;
      
      //----------------------------------------------------------------------
      //!  Sets and returns the contained algorithm.
      //----------------------------------------------------------------------
      uint8_t Algorithm(uint8_t algorithm);
      
      //----------------------------------------------------------------------
      //!  Returns the contained labels.
      //----------------------------------------------------------------------
      uint8_t Labels() const;
      
      //----------------------------------------------------------------------
      //!  Sets and returns the contained labels.
      //----------------------------------------------------------------------
      uint8_t Labels(uint8_t labels);
      
      //----------------------------------------------------------------------
      //!  Returns the contained original TTL.
      //----------------------------------------------------------------------
      uint32_t OriginalTTL() const;
      
      //----------------------------------------------------------------------
      //!  Sets and returns the contained original TTL.
      //----------------------------------------------------------------------
      uint32_t OriginalTTL(uint32_t originalTTL);
      
      //----------------------------------------------------------------------
      //!  Returns the contained signature expiration.
      //----------------------------------------------------------------------
      uint32_t SignatureExpiration() const;
      
      //----------------------------------------------------------------------
      //!  Sets and returns the contained signature expiration.
      //----------------------------------------------------------------------
      uint32_t SignatureExpiration(uint32_t signatureExpiration);
      
      //----------------------------------------------------------------------
      //!  Returns the contained signature inception.
      //----------------------------------------------------------------------
      uint32_t SignatureInception() const;
      
      //----------------------------------------------------------------------
      //!  Sets and returns the contained signature inception.
      //----------------------------------------------------------------------
      uint32_t SignatureInception(uint32_t signatureInception);
      
      //----------------------------------------------------------------------
      //!  Returns the contained key tag.
      //----------------------------------------------------------------------
      uint16_t KeyTag() const;
      
      //----------------------------------------------------------------------
      //!  Sets and returns the contained key tag.
      //----------------------------------------------------------------------
      uint16_t KeyTag(uint16_t keyTag);
      
      //----------------------------------------------------------------------
      //!  Returns the contained signer's name.
      //----------------------------------------------------------------------
      const std::string & SignersName() const;
      
      //----------------------------------------------------------------------
      //!  Sets and returns the contained signer's name.
      //----------------------------------------------------------------------
      const std::string & SignersName(const std::string & signersName);
      
      //----------------------------------------------------------------------
      //!  Returns the contained signature.
      //----------------------------------------------------------------------
      const std::string & Signature() const;
      
      //----------------------------------------------------------------------
      //!  Sets and returns the contained signature.
      //----------------------------------------------------------------------
      const std::string & Signature(const std::string & signature);

      //----------------------------------------------------------------------
      //!  Encodes the RRSIG resource record data into buffer @ pkt of length
      //!  @c pktlen, starting at @c ptr.  Returns the address immediately
      //!  following the encoded RRSIG resource record data in @c pkt on
      //!  success.  Throws std::out_of_range if @c pkt is too short to
      //!  contain the encoded RRSIG resource record data.
      //----------------------------------------------------------------------
      uint8_t *Encode(uint8_t *pkt, uint8_t *ptr, uint16_t pktlen,
                      LabelPositions & lps) const;
      
      //----------------------------------------------------------------------
      //!  Decodes the RRSIG resource record data from buffer @c pkt of length
      //!  @c pktlen, starting at @c ptr.  Returns the address immediately
      //!  following the encoded  RRSIG resource record data in @c pkt on
      //!  success.  Throws std::out_of_range if @c pkt was too short to
      //!  contain an encoded RRSIG resource record data.
      //----------------------------------------------------------------------
      const uint8_t *Decode(const uint8_t *pkt, const uint8_t *ptr,
                            uint16_t pktlen, uint16_t rdlen);
      
      //----------------------------------------------------------------------
      //!  Print the RRSIG resource record data to an ostream in human
      //!  readable form.
      //----------------------------------------------------------------------
      friend std::ostream & operator << (std::ostream & os,
                                         const RRDataRRSIG & rrsig);

      //----------------------------------------------------------------------
      //!  Returns the mnemonic for the given algorithm @c algo.  If @c algo
      //!  is an algorithm we do not recognize, returns a string
      //!  representation of @c algo.
      //----------------------------------------------------------------------
      static std::string AlgorithmName(uint8_t algo);
      
    private:
      typedef struct __attribute__((packed)) {
        uint16_t   typeCovered;
        uint8_t    algorithm;
        uint8_t    labels;
        uint32_t   originalTTL;
        uint32_t   signatureExpiration;
        uint32_t   signatureInception;
        uint16_t   keyTag;
      } fixed_data_t;

      fixed_data_t  _fixedData;
      std::string   _signersName;
      std::string   _signature;

      static fixed_data_t ToNetworkByteOrder(const fixed_data_t *data);
      static fixed_data_t ToHostByteOrder(const fixed_data_t *data);
    };
    
  }  // namespace Dns

}  // namespace Dwm

#endif  // _DWMDNSRRDATARRSIG_HH_
