%{
//===========================================================================
// @(#) $Name:$
// @(#) $Id: DwmDnsEtcHostsParser.yy 11521 2020-12-16 23:03:44Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2018
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmDnsEtcHostsParser.yy
//!  \brief Dwm::Dns::EtcHosts class implementation and /etc/hosts parser
//---------------------------------------------------------------------------

  extern int etchostslex();

  #include <cstring>
  #include <mutex>
  #include <utility>
  
  #include "DwmSvnTag.hh"
  #include "DwmDnsEtcHosts.hh"
  #include "DwmDnsUtils.hh"
  
  static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/DwmDns/tags/DwmDns-0.2.5/classes/src/DwmDnsEtcHostsParser.yy 11521 $");

  extern "C" {
    extern int etchostserror(const char *);
  }

  extern FILE  *etchostsin;

  static Dwm::Dns::EtcHosts  *g_etcHosts = 0;
%}

%define api.prefix {etchosts}

%union {
  const std::string                         *stringVal;
  const std::pair<std::string,std::string>  *stringPairVal;
}

%token<stringVal>  IPADDR NAME

%type<stringPairVal> Line

%%

Lines: Line
{
  delete $1;
}
| Lines Line
{
  delete $2;
};

Line: IPADDR NAME
{
  $$ = new std::pair<std::string,std::string>(*($1), *($2));
  g_etcHosts->Add(*($1), *($2));
  delete $1;
  delete $2;
}
| Line NAME
{
  $$ = new std::pair<std::string,std::string>(($1)->first, *($2));
  g_etcHosts->Add(($1)->first, *($2));
  delete $1;
  delete $2;
};

%%

using namespace std;

namespace Dwm {

  namespace Dns {

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    EtcHosts::EtcHosts(const string & hostsfile)
        : _namesToV4Addrs(), _namesToV6Addrs(), _v6addrsToNames(),
          _v4addrsToNames()
    {
      static std::mutex  mtx;
      std::lock_guard<std::mutex>  lck(mtx);
      g_etcHosts = this;
      etchostsin = fopen(hostsfile.c_str(), "r");
      if (etchostsin) {
        if (etchostsparse() != 0) {
          cerr << "Failed to parse hosts file '" << hostsfile << "'\n";
        }
        fclose(etchostsin);
      }
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void EtcHosts::Add(const string & ipAddr, const string & name)
    {
      in6_addr  in6Addr;
      in_addr   inAddr;
      if (inet_pton(AF_INET6, ipAddr.c_str(), &in6Addr) == 1) {
        _namesToV6Addrs[name] = in6Addr;
        _v6addrsToNames.insert(pair<in6_addr,string>(in6Addr, name));
      }
      else if (inet_pton(AF_INET, ipAddr.c_str(), &inAddr) == 1) {
        _namesToV4Addrs[name] = inAddr;
        _v4addrsToNames.insert(pair<in_addr,string>(inAddr, name));
      }
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool EtcHosts::Get(const std::string & name, uint16_t rrtype,
                       std::vector<ResourceRecord> & results)
    {
      results.clear();
      in6_addr  in6Addr;
      in_addr   inAddr;
      if (rrtype == ResourceRecord::k_typePTR) {
        if (inet_pton(AF_INET6, name.c_str(), &in6Addr) == 1) {
          auto  its = _v6addrsToNames.equal_range(in6Addr);
          for (auto it = its.first; it != its.second; ++it) {
            ResourceRecord  rr;
            string  rrname;
            if (ToArpa(name, rrname)) {
              rr.Name(rrname);
              rr.TTL(0xFFFFFFFF);
              RRDataPTR  ptr(it->second);
              rr.Data(ptr);
              results.push_back(rr);
            }
          }
        }
        else if (inet_pton(AF_INET, name.c_str(), &inAddr) == 1) {
          auto  its = _v4addrsToNames.equal_range(inAddr);
          for (auto it = its.first; it != its.second; ++it) {
            ResourceRecord  rr;
            string  rrname;
            if (ToArpa(name, rrname)) {
              rr.Name(rrname);
              rr.TTL(0xFFFFFFFF);
              rr.Data(RRDataPTR(it->second));
              results.push_back(rr);
            }
          }
        }
      }
      else if (rrtype == ResourceRecord::k_typeAAAA) {
        auto  it6 = _namesToV6Addrs.find(name);
        if (it6 != _namesToV6Addrs.end()) {
          ResourceRecord  rr;
          rr.Name(name);
          rr.TTL(0xFFFFFFFF);
          rr.Data(RRDataAAAA(it6->second));
          results.push_back(rr);
        }
      }
      else if (rrtype == ResourceRecord::k_typeA) {
        auto  it4 = _namesToV4Addrs.find(name);
        if (it4 != _namesToV4Addrs.end()) {
          ResourceRecord  rr;
          rr.Name(name);
          rr.TTL(0xFFFFFFFF);
          rr.Data(RRDataA(it4->second));
          results.push_back(rr);
        }
      }
      return (! results.empty());
    }
    
      
  }  // namespace Dns

}  // namespace Dwm
