//===========================================================================
// @(#) $DwmPath: dwm/DwmDns/tags/DwmDns-0.2.5/classes/src/DwmDnsNameServer.cc 12264 $
// @(#) $Id: DwmDnsNameServer.cc 12264 2023-10-27 02:14:08Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2017, 2018
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DnsNameServer.cc
//!  \brief Dwm::Dns::NameServer class implementation
//---------------------------------------------------------------------------

extern "C" {
  #include <sys/select.h>
  #include <sys/types.h>
  #include <sys/socket.h>
  #include <netinet/in.h>
  #include <arpa/inet.h>
}

#include <cassert>
#include <cstring>
#include <stdexcept>
#include <utility>

#include "DwmSvnTag.hh"
#include "DwmSysLogger.hh"
#include "DwmDnsNameServer.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/DwmDns/tags/DwmDns-0.2.5/classes/src/DwmDnsNameServer.cc 12264 $");

using namespace std;

namespace Dwm {

  namespace Dns {

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    NameServer::NameServer()
        : _addrFamily(AF_INET), _udpfd(-1), _tcpfd(-1)
    {
      memset(&_addr, 0, sizeof(_addr));      
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    NameServer::NameServer(NameServer && ns) noexcept
        : _udpfd(std::exchange(ns._udpfd, -1)),
          _tcpfd(std::exchange(ns._tcpfd, -1)),
          _addrFamily(std::exchange(ns._addrFamily, AF_INET))
    {
      memcpy(&_addr, &ns._addr, sizeof(_addr));
      memset(&ns._addr, 0, sizeof(ns._addr));
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    NameServer & NameServer::operator = (NameServer && ns) noexcept
    {
      if (0 <= _udpfd) {
        close(_udpfd);
      }
      _udpfd = std::exchange(ns._udpfd, -1);
      if (0 <= _tcpfd) {
        close(_tcpfd);
      }
      _tcpfd = std::exchange(ns._tcpfd, -1);
      
      _addrFamily = std::exchange(ns._addrFamily, AF_INET);
      memcpy(&_addr, &ns._addr, sizeof(_addr));
      memset(&ns._addr, 0, sizeof(ns._addr));
      
      return *this;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    NameServer::NameServer(const struct in_addr & addr)
        : _addrFamily(AF_INET), _udpfd(-1), _tcpfd(-1)
    {
      memset(&_addr, 0, sizeof(_addr));
      _addr.in4Addr.sin_family = AF_INET;
      _addr.in4Addr.sin_addr = addr;
      _addr.in4Addr.sin_port = htons(53);
#ifndef __linux__
      _addr.in4Addr.sin_len = sizeof(_addr.in4Addr);
#endif
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    NameServer::NameServer(const Ipv4Address & addr)
        : _addrFamily(AF_INET), _udpfd(-1), _tcpfd(-1)
    {
      memset(&_addr, 0, sizeof(_addr));
      _addr.in4Addr.sin_family = AF_INET;
      _addr.in4Addr.sin_addr.s_addr = addr.Raw();
      _addr.in4Addr.sin_port = htons(53);
#ifndef __linux__
      _addr.in4Addr.sin_len = sizeof(_addr.in4Addr);
#endif
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    NameServer::NameServer(const struct in6_addr & addr)
        : _addrFamily(AF_INET6), _udpfd(-1), _tcpfd(-1)
    {
      memset(&_addr, 0, sizeof(_addr));
      _addr.in6Addr.sin6_family = AF_INET6;
      _addr.in6Addr.sin6_addr = addr;
      _addr.in6Addr.sin6_port = htons(53);
#ifndef __linux__
      _addr.in6Addr.sin6_len = sizeof(struct sockaddr_in6);
#endif
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    NameServer::NameServer(const std::string & addr)
        : _udpfd(-1), _tcpfd(-1)
    {
      memset(&_addr, 0, sizeof(_addr));
      in6_addr  in6Addr;
      if (inet_pton(AF_INET6, addr.c_str(), &in6Addr) == 1) {
        _addr.in6Addr.sin6_family = AF_INET6;
        _addr.in6Addr.sin6_addr = in6Addr;
        _addr.in6Addr.sin6_port = htons(53);
#ifndef __linux__
        _addr.in6Addr.sin6_len = sizeof(struct sockaddr_in6);
#endif
        _addrFamily = AF_INET6;
      }
      else {
        in_addr  inAddr;
        if (inet_pton(AF_INET, addr.c_str(), &inAddr) == 1) {
          _addr.in4Addr.sin_family = AF_INET;
          _addr.in4Addr.sin_addr = inAddr;
          _addr.in4Addr.sin_port = htons(53);
#ifndef __linux__
          _addr.in4Addr.sin_len = sizeof(_addr.in4Addr);
#endif
          _addrFamily = AF_INET;
        }
        else {
          Syslog(LOG_ERR, "invalid address for NameServer %s", addr.c_str());
          throw std::invalid_argument("invalid address for NameServer");
        }
      }
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    NameServer::~NameServer()
    {
      if (_udpfd >= 0) {
        close(_udpfd);
      }
      if (_tcpfd >= 0) {
        close(_tcpfd);
      }
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    tuple<sa_family_t,const struct sockaddr *,socklen_t>
    NameServer::Address() const
    {
      tuple<sa_family_t,const struct sockaddr *,socklen_t>  rc;
      get<0>(rc) = _addrFamily;
      if (_addrFamily == AF_INET6) {
        get<1>(rc) = (const struct sockaddr *)&_addr.in6Addr;
        get<2>(rc) = sizeof(_addr.in6Addr);
      }
      else {
        get<1>(rc) = (const struct sockaddr *)&_addr.in4Addr;
        get<2>(rc) = sizeof(_addr.in4Addr);
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool NameServer::SendMessage(const Message & message)
    {
      if (_udpfd < 0) {
        if (! Open()) {
          return false;
        }
      }
      AddrInfo  addrInfo = Address();
      return (message.SendTo(_udpfd, 0, get<1>(addrInfo), get<2>(addrInfo))
              > 0);
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool NameServer::WriteMessage(const Message & message)
    {
      bool  rc = false;
      if (_tcpfd < 0) {
        if (! Open(true)) {
          return false;
        }
      }
      return (message.Write(_tcpfd) > 0);
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool NameServer::AnswerWaiting(const struct timeval & timeout) const
    {
      bool  rc = false;
      if (_udpfd >= 0) {
        struct timeval  tv = timeout;
        fd_set readFdSet;
        FD_ZERO(&readFdSet);
        FD_SET(_udpfd, &readFdSet);
        if (select(_udpfd + 1, &readFdSet, 0, 0, &tv) > 0) {
          if (FD_ISSET(_udpfd, &readFdSet)) {
            rc = true;
          }
        }
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool NameServer::ReceiveMessage(Message & message)
    {
      bool  rc = false;
      
      struct timeval  timeout = { 2, 0 };
      if (AnswerWaiting(timeout)) {
        socklen_t  fromLen;
        AddrInfo   addrInfo = Address();
        if (get<0>(addrInfo) == AF_INET) {
          struct sockaddr_in  from;
          fromLen = sizeof(from);
          if (message.RecvFrom(_udpfd, 0, (struct sockaddr *)&from, &fromLen)
              > 0) {
            rc = true;
          }
        }
        else {
          struct sockaddr_in6  from;
          fromLen = sizeof(from);
          if (message.RecvFrom(_udpfd, 0, (struct sockaddr *)&from, &fromLen)
              > 0) {
            rc = true;
          }
        }
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool NameServer::ReadMessage(Message & message, int32_t timeout)
    {
      assert(timeout >= 0);
      bool  rc = false;
      if (_tcpfd >= 0) {
        struct timeval  tv = { timeout / 1000, (timeout % 1000) * 1000 };
        fd_set readFdSet;
        FD_ZERO(&readFdSet);
        FD_SET(_tcpfd, &readFdSet);
        if (select(_tcpfd + 1, &readFdSet, 0, 0, &tv) > 0) {
          if (FD_ISSET(_tcpfd, &readFdSet)) {
            rc = (message.Read(_tcpfd) > 0);
            if (! rc) {
              close(_tcpfd);
              _tcpfd = -1;
            }
          }
        }
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool NameServer::Open(bool tcp)
    {
      if (! tcp) {
        _udpfd = socket(_addrFamily, SOCK_DGRAM, 0);
        if (_udpfd < 0) {
          Syslog(LOG_ERR, "socket() failed: %m");
        }
        return (_udpfd >= 0);
      }
      else {
        _tcpfd = socket(_addrFamily, SOCK_STREAM, 0);
        if (_tcpfd >= 0) {
          if (_addrFamily == AF_INET) {
            if (connect(_tcpfd, (const struct sockaddr *)&_addr.in4Addr,
                        sizeof(_addr.in4Addr)) == 0) {
              return true;
            }
            else {
              char  buf[INET_ADDRSTRLEN];
              Syslog(LOG_ERR, "connect(%d, %s) failed: %m",
                     _tcpfd,
                     inet_ntop(AF_INET, &_addr.in4Addr, buf, sizeof(buf)));
              close(_tcpfd);
              _tcpfd = -1;
              return false;
            }
          }
          else {
            if (connect(_tcpfd, (const struct sockaddr *)&_addr.in6Addr,
                        sizeof(_addr.in6Addr)) == 0) {
              return true;
            }
            else {
              char  buf[INET6_ADDRSTRLEN];
              Syslog(LOG_ERR, "connect(%d, %s) failed: %m",
                     _tcpfd,
                     inet_ntop(AF_INET6, &_addr.in4Addr, buf, sizeof(buf)));
              close(_tcpfd);
              _tcpfd = -1;
              return false;
            }
          }
        }
        else {
          Syslog(LOG_ERR, "socket() failed: %m");
          return false;
        }
      }
    }
    
  }  // namespace Dns

}  // namespace Dwm

