//===========================================================================
// @(#) $DwmPath: dwm/DwmDns/tags/DwmDns-0.2.5/classes/src/DwmDnsRRDataDS.cc 10133 $
// @(#) $Id: DwmDnsRRDataDS.cc 10133 2018-01-27 17:41:32Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2018
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmDnsRRDataDS.cc
//!  \brief Dwm::Dns::RRDataDS class implementation
//---------------------------------------------------------------------------

extern "C" {
  #include <arpa/inet.h>
}

#include <cstring>
#include <iomanip>
#include <sstream>

#include "DwmSvnTag.hh"
#include "DwmBase64.hh"
#include "DwmDnsRRDataDS.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/DwmDns/tags/DwmDns-0.2.5/classes/src/DwmDnsRRDataDS.cc 10133 $");

using namespace std;

namespace Dwm {

  namespace Dns {

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    RRDataDS::RRDataDS()
        : _keyTag(0), _algorithm(0), _digestType(0), _digest()
    {}
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    RRDataDS::RRDataDS(uint16_t keyTag, uint8_t algorithm, uint8_t digestType,
                       const std::string & digest)
        : _keyTag(keyTag), _algorithm(algorithm), _digestType(digestType),
          _digest(digest)
    {}
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool RRDataDS::operator == (const RRDataDS & ds) const
    {
      return ((_keyTag == ds._keyTag)
              && (_algorithm == ds._algorithm)
              && (_digestType == ds._digestType)
              && (_digest == ds._digest));
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint16_t RRDataDS::KeyTag() const
    {
      return _keyTag;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint16_t RRDataDS::KeyTag(uint16_t keyTag)
    {
      _keyTag = keyTag;
      return _keyTag;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint8_t RRDataDS::Algorithm() const
    {
      return _algorithm;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint8_t RRDataDS::Algorithm(uint8_t algorithm)
    {
      _algorithm = algorithm;
      return _algorithm;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint8_t RRDataDS::DigestType() const
    {
      return _digestType;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint8_t RRDataDS::DigestType(uint16_t digestType)
    {
      _digestType = digestType;
      return _digestType;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const std::string & RRDataDS::Digest() const
    {
      return _digest;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const std::string & RRDataDS::Digest(const std::string & digest)
    {
      _digest = digest;
      return _digest;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint8_t *RRDataDS::Encode(uint8_t *pkt, uint8_t *ptr,
                              uint16_t pktlen, LabelPositions & lps) const
    {
      if ((ptr + 4) <= (pkt + pktlen)) {
        uint16_t  val = htons(_keyTag);
        memcpy(ptr, &val, sizeof(val));
        ptr += sizeof(val);
        *ptr++ = _algorithm;
        *ptr++ = _digestType;
        if (_digest.size() > 0) {
          if ((ptr + _digest.size()) <= (pkt + pktlen)) {
            memcpy(ptr, _digest.c_str(), _digest.size());
            ptr += _digest.size();
          }
          else {
            throw out_of_range("Dwm::Dns::RRDataDS will not fit in packet");
          }
        }
      }
      else {
        throw out_of_range("Dwm::Dns::RRDataDS will not fit in packet");
      }
      return ptr;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const uint8_t *RRDataDS::Decode(const uint8_t *pkt, const uint8_t *ptr,
                                    uint16_t pktlen, uint16_t rdlen)
    {
      _keyTag = 0;
      _algorithm = 0;
      _digestType = 0;
      _digest.clear();

      uint16_t        bytesLeft = rdlen;
      const uint8_t  *origptr = ptr;
      if ((ptr + 4) <= (pkt + pktlen)) {
        memcpy(&_keyTag, ptr, sizeof(_keyTag));
        _keyTag = ntohs(_keyTag);
        ptr += sizeof(_keyTag);
        _algorithm = *ptr++;
        _digestType = *ptr++;
        bytesLeft -= (ptr - origptr);
        if (bytesLeft) {
          _digest.assign((const char *)ptr, bytesLeft);
          ptr += bytesLeft;
        }
      }
      return ptr;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    ostream & operator << (ostream & os, const RRDataDS & ds)
    {
      if (os) {
        os << ds._keyTag << ' ' << (uint16_t)ds._algorithm << ' '
           << (uint16_t)ds._digestType << ' ';
        ostringstream  oss;
        oss << hex << setfill('0');
        for (uint8_t c : ds._digest) {
          oss << setw(2) << (uint16_t)c;
        }
        os << oss.str();
      }
      return os;
    }
    
  }  // namespace Dns

}  // namespace Dwm
