//===========================================================================
// @(#) $DwmPath: dwm/DwmDns/tags/DwmDns-0.2.5/classes/src/DwmDnsRRDataLOC.cc 10138 $
// @(#) $Id: DwmDnsRRDataLOC.cc 10138 2018-01-28 07:39:44Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2000, 2016, 2018
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmDnsRRDataLOC.cc
//!  \brief Dwm::Dns::RRDataLOC class implementation
//---------------------------------------------------------------------------

extern "C" {
  #include <arpa/inet.h>
}

#include <cstring>
#include <iomanip>
#include <stdexcept>

#include "DwmSvnTag.hh"
#include "DwmDnsRRDataLOC.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/DwmDns/tags/DwmDns-0.2.5/classes/src/DwmDnsRRDataLOC.cc 10138 $");

using namespace std;

namespace Dwm {

  namespace Dns {

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    static const char *precsize_ntoa(uint8_t prec)
    {
      unsigned int poweroften[10] = {1, 10, 100, 1000, 10000, 100000,
                                     1000000,10000000,100000000,1000000000 };
      
      static char retbuf[sizeof("90000000.00")];
      unsigned long val;
      int mantissa, exponent;
      
      mantissa = (int)((prec >> 4) & 0x0f) % 10;
      exponent = (int)((prec >> 0) & 0x0f) % 10;
      
      val = mantissa * poweroften[exponent];
      
      (void) sprintf(retbuf,"%ld.%.2ld", val/100, val%100);
      return (retbuf);
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    RRDataLOC::RRDataLOC()
    {
      memset(&_data, 0, sizeof(_data));
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    RRDataLOC::RRDataLOC(uint8_t version, uint8_t size,
                         uint8_t horizontalPrecision,
                         uint8_t verticalPrecision,
                         uint32_t latitude, uint32_t longitude,
                         uint32_t altitude)
    {
      _data.version = version;
      _data.size = size;
      _data.horizontalPrecision = horizontalPrecision;
      _data.verticalPrecision = verticalPrecision;
      _data.latitude = latitude;
      _data.longitude = longitude;
      _data.altitude = altitude;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool RRDataLOC::operator == (const RRDataLOC & loc) const
    {
      return (memcmp(&_data, &loc._data, sizeof(_data)) == 0);
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint8_t RRDataLOC::Version() const
    {
      return _data.version;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint8_t RRDataLOC::Version(uint8_t version)
    {
      _data.version = version;
      return _data.version;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint8_t RRDataLOC::Size() const
    {
      return _data.size;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint8_t RRDataLOC::Size(uint8_t size)
    {
      _data.size = size;
      return _data.size;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint8_t RRDataLOC::HorizontalPrecision() const
    {
      return _data.horizontalPrecision;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint8_t RRDataLOC::HorizontalPrecision(uint8_t horizontalPrecision)
    {
      _data.horizontalPrecision = horizontalPrecision;
      return _data.horizontalPrecision;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint8_t RRDataLOC::VerticalPrecision() const
    {
      return _data.verticalPrecision;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint8_t RRDataLOC::VerticalPrecision(uint8_t verticalPrecision)
    {
      _data.verticalPrecision = verticalPrecision;
      return _data.verticalPrecision;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint32_t RRDataLOC::Latitude() const
    {
      return _data.latitude;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint32_t RRDataLOC::Latitude(uint32_t latitude)
    {
      _data.latitude = latitude;
      return _data.latitude;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint32_t RRDataLOC::Longitude() const
    {
      return _data.longitude;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint32_t RRDataLOC::Longitude(uint32_t longitude)
    {
      _data.longitude = longitude;
      return _data.longitude;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint32_t RRDataLOC::Altitude() const
    {
      return _data.altitude;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint32_t RRDataLOC::Altitude(uint32_t altitude)
    {
      _data.altitude = altitude;
      return _data.altitude;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint8_t *RRDataLOC::Encode(uint8_t *pkt, uint8_t *ptr,
                               uint16_t pktlen, LabelPositions & lps) const
    {
      if ((ptr + sizeof(_data)) <= (pkt + pktlen)) {
        data_t  data = ToNetworkByteOrder(&_data);
        memcpy(ptr, &data, sizeof(data));
        ptr += sizeof(_data);
      }
      else {
        throw out_of_range("Dwm::Dns::RRDataLOC will not fit in packet");
      }
      return ptr;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const uint8_t *RRDataLOC::Decode(const uint8_t *pkt, const uint8_t *ptr,
                                     uint16_t pktlen, uint16_t rdlen)
    {
      memset(&_data, 0, sizeof(_data));
      if ((ptr + sizeof(_data)) <= (pkt + pktlen)) {
        data_t  data;
        memcpy(&data, ptr, sizeof(data));
        ptr += sizeof(_data);
        _data = ToHostByteOrder(&data);
      }
      else {
        throw out_of_range("packet too short to contain Dwm::Dns::RRDataLOC");
      }
      return ptr;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    ostream & operator << (ostream & os, const RRDataLOC & loc)
    {
      char          tmpbuf[255*3];
      char         *cp;
      const u_char *rcp;
      
      int latdeg, latmin, latsec, latsecfrac;
      int longdeg, longmin, longsec, longsecfrac;
      char northsouth, eastwest;
      int altmeters, altfrac, altsign;
      
      const int referencealt = 100000 * 100;
      
      int32_t latval, longval, altval;
      uint32_t templ;
      uint8_t sizeval, hpval, vpval;
      
      cp = tmpbuf;
      
      if (loc.Version() != 0) {
        os << "; error: unknown LOC RR version";
        return(os);
      }
      
      sizeval = loc.Size();
      
      hpval = loc.HorizontalPrecision();
      vpval = loc.VerticalPrecision();
      
      latval = (loc.Latitude() - ((unsigned)1<<31));
      
      longval = (loc.Longitude() - ((unsigned)1<<31));
      
      templ = loc.Altitude();
      if (templ < referencealt) { /* below WGS 84 spheroid */
        altval = referencealt - templ;
        altsign = -1;
      } else {
        altval = templ - referencealt;
        altsign = 1;
      }
      
      if (latval < 0) {
        northsouth = 'S';
        latval = -latval;
      }
      else
        northsouth = 'N';
      
      latsecfrac = latval % 1000;
      latval = latval / 1000;
      latsec = latval % 60;
      latval = latval / 60;
      latmin = latval % 60;
      latval = latval / 60;
      latdeg = latval;
      
      if (longval < 0) {
        eastwest = 'W';
        longval = -longval;
      }
      else
        eastwest = 'E';
      
      longsecfrac = longval % 1000;
      longval = longval / 1000;
      longsec = longval % 60;
      longval = longval / 60;
      longmin = longval % 60;
      longval = longval / 60;
      longdeg = longval;
      
      altfrac = altval % 100;
      altmeters = (altval / 100) * altsign;
      
      os << latdeg << " " << latmin << " " << latsec << "." << setfill('0')
         << setw(3) << latsecfrac << setfill(' ')
         << " " << northsouth << " " << longdeg << " " << longmin << " "
         << longsec << "." << setfill('0') << setw(3) << longsecfrac
         << setfill(' ') << " " << eastwest << " "
         << altmeters << "." << setfill('0') << setw(2) << altfrac << 'm'
         << setfill(' ') << " " << precsize_ntoa(sizeval) << "m ";
      os << precsize_ntoa(hpval) << "m ";
      os << precsize_ntoa(vpval) << "m";
      
      return(os);
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    RRDataLOC::data_t
    RRDataLOC::ToNetworkByteOrder(const RRDataLOC::data_t *data)
    {
      RRDataLOC::data_t  rc = *data;
      rc.latitude = htonl(rc.latitude);
      rc.longitude = htonl(rc.longitude);
      rc.altitude = htonl(rc.altitude);
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    RRDataLOC::data_t
    RRDataLOC::ToHostByteOrder(const RRDataLOC::data_t *data)
    {
      RRDataLOC::data_t  rc = *data;
      rc.latitude = ntohl(rc.latitude);
      rc.longitude = ntohl(rc.longitude);
      rc.altitude = ntohl(rc.altitude);
      return rc;
    }
    

    
  }  // namespace Dns

}  // namespace Dwm
