//===========================================================================
// @(#) $Name:$
// @(#) $Id: TestDnsNameServer.cc 10734 2020-05-07 18:13:40Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2018
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file TestDnsNameServer.cc
//!  \brief NOT YET DOCUMENTED
//---------------------------------------------------------------------------

extern "C" {
  #include <sys/types.h>
  #include <sys/socket.h>
  #include <netinet/in.h>
  #include <arpa/inet.h>
}

#include "DwmSvnTag.hh"
#include "DwmUnitAssert.hh"
#include "DwmDnsNameServer.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/DwmDns/tags/DwmDns-0.2.5/classes/tests/TestDnsNameServer.cc 10734 $");

using namespace std;
using namespace Dwm;

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
void TestGoogleA()
{
  Dns::Message  message;
  message.Header().RecursionDesired(true);
  Dns::MessageQuestion  question("www.google.com",
                                 Dns::MessageQuestion::k_typeA,
                                 Dns::MessageQuestion::k_classIN);
  message.Questions().push_back(question);

  Dns::NameServer  nameServer(Ipv4Address("8.8.8.8"));
  if (UnitAssert(nameServer.SendMessage(message) > 0)) {
    UnitAssert(nameServer.ReceiveMessage(message));
    UnitAssert(message.Questions().size() > 0);
    UnitAssert(message.Answers().size() > 0);
    for (auto & answer : message.Answers()) {
      UnitAssert(answer.Name() == "www.google.com");
      Dns::RRDataA  *a = answer.Data<Dns::RRDataA>();
      if (UnitAssert(a != nullptr)) {
        UnitAssert(a->InAddr().s_addr != INADDR_NONE);
      }
    }
  }
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
void TestGoogleAAAA()
{
  Dns::Message  message;
  message.Header().RecursionDesired(true);
  Dns::MessageQuestion  question("www.google.com",
                                 Dns::MessageQuestion::k_typeAAAA,
                                 Dns::MessageQuestion::k_classIN);
  message.Questions().push_back(question);

  Dns::NameServer  nameServer(Ipv4Address("8.8.8.8"));
  if (UnitAssert(nameServer.SendMessage(message) > 0)) {
    UnitAssert(nameServer.ReceiveMessage(message));
    UnitAssert(message.Questions().size() > 0);
    UnitAssert(message.Answers().size() > 0);
    for (auto & answer : message.Answers()) {
      UnitAssert(answer.Name() == "www.google.com");
      Dns::RRDataAAAA  *aaaa = answer.Data<Dns::RRDataAAAA>();
      UnitAssert(aaaa != nullptr);
    }
  }
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
void TestGoogleSOA()
{
  Dns::Message  message;
  message.Header().RecursionDesired(true);
  Dns::MessageQuestion  question("google.com",
                                 Dns::MessageQuestion::k_typeSOA,
                                 Dns::MessageQuestion::k_classIN);
  message.Questions().push_back(question);

  Dns::NameServer  nameServer(Ipv4Address("8.8.8.8"));
  if (UnitAssert(nameServer.SendMessage(message) > 0)) {
    UnitAssert(nameServer.ReceiveMessage(message));
    UnitAssert(message.Questions().size() > 0);
    UnitAssert(message.Answers().size() > 0);
    for (auto & answer : message.Answers()) {
      UnitAssert(answer.Name() == "google.com");
      Dns::RRDataSOA  *soa = answer.Data<Dns::RRDataSOA>();
      if (UnitAssert(soa != nullptr)) {
        UnitAssert(! soa->Mname().empty());
        UnitAssert(! soa->Rname().empty());
      }
    }
  }
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
void TestGooglePTR()
{
  Dns::Message  message;
  message.Header().RecursionDesired(true);
  Dns::MessageQuestion  question("8.8.8.8.in-addr.arpa",
                                 Dns::MessageQuestion::k_typePTR,
                                 Dns::MessageQuestion::k_classIN);
  message.Questions().push_back(question);

  Dns::NameServer  nameServer(Ipv4Address("8.8.8.8"));
  if (UnitAssert(nameServer.SendMessage(message) > 0)) {
    UnitAssert(nameServer.ReceiveMessage(message));
    UnitAssert(message.Questions().size() > 0);
    UnitAssert(message.Answers().size() > 0);
    for (auto & answer : message.Answers()) {
      Dns::RRDataPTR  *ptr = answer.Data<Dns::RRDataPTR>();
      UnitAssert(answer.Name() == "8.8.8.8.in-addr.arpa");
      if (UnitAssert(ptr != nullptr)) {
        UnitAssert(ptr->Ptr().find("dns.google") != string::npos);
      }
    }
  }
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
void TestTwitCNAME()
{
  Dns::Message  message;
  message.Header().RecursionDesired(true);
  Dns::MessageQuestion  question("www.twit.tv",
                                 Dns::MessageQuestion::k_typeCNAME,
                                 Dns::MessageQuestion::k_classIN);
  message.Questions().push_back(question);

  Dns::NameServer  nameServer(Ipv4Address("8.8.8.8"));
  if (UnitAssert(nameServer.SendMessage(message) > 0)) {
    UnitAssert(nameServer.ReceiveMessage(message));
    UnitAssert(message.Questions().size() > 0);
    UnitAssert(message.Answers().size() > 0);
    for (auto & answer : message.Answers()) {
      Dns::RRDataCNAME  *cname = answer.Data<Dns::RRDataCNAME>();
      UnitAssert(answer.Name() == "www.twit.tv");
      if (UnitAssert(cname != nullptr)) {
        UnitAssert(! cname->Cname().empty());
      }
    }
  }
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
void TestGRCCAA()
{
  Dns::Message  message;
  message.Header().RecursionDesired(true);
  Dns::MessageQuestion  question("grc.com",
                                 Dns::MessageQuestion::k_typeCAA,
                                 Dns::MessageQuestion::k_classIN);
  message.Questions().push_back(question);

  Dns::NameServer  nameServer(Ipv4Address("8.8.8.8"));
  if (UnitAssert(nameServer.SendMessage(message) > 0)) {
    UnitAssert(nameServer.ReceiveMessage(message));
    UnitAssert(message.Questions().size() > 0);
    UnitAssert(message.Answers().size() > 0);
    for (auto & answer : message.Answers()) {
      UnitAssert(answer.Name() == "grc.com");
      Dns::RRDataCAA  *caa = answer.Data<Dns::RRDataCAA>();
      if (UnitAssert(caa != nullptr)) {
        UnitAssert(caa->Tag() == "issue");
        UnitAssert(caa->Value() == "digicert.com");
      }
    }
  }
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
void TestIETFDNSKEY()
{
  Dns::Message  message;
  message.Header().RecursionDesired(true);
  Dns::MessageQuestion  question("ietf.org",
                                 Dns::MessageQuestion::k_typeDNSKEY,
                                 Dns::MessageQuestion::k_classIN);
  message.Questions().push_back(question);
  message.EnableEDNS(4096, false);
  
  Dns::NameServer  nameServer(Ipv4Address("8.8.8.8"));
  if (UnitAssert(nameServer.SendMessage(message) > 0)) {
    UnitAssert(nameServer.ReceiveMessage(message));
    UnitAssert(message.Questions().size() > 0);
    UnitAssert(message.Answers().size() > 0);
    for (auto & answer : message.Answers()) {
      UnitAssert(answer.Name() == "ietf.org");
      Dns::RRDataDNSKEY  *dnskey = answer.Data<Dns::RRDataDNSKEY>();
      if (UnitAssert(dnskey != nullptr)) {
        UnitAssert(! dnskey->PublicKey().empty());
      }
    }
  }
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
void TestIETFNSEC()
{
  Dns::Message  message;
  message.Header().RecursionDesired(true);
  Dns::MessageQuestion  question("ietf.org",
                                 Dns::MessageQuestion::k_typeNSEC,
                                 Dns::MessageQuestion::k_classIN);
  message.Questions().push_back(question);
  //  message.EnableEDNS(4096, false);
  
  Dns::NameServer  nameServer(Ipv4Address("8.8.8.8"));
  if (UnitAssert(nameServer.SendMessage(message) > 0)) {
    UnitAssert(nameServer.ReceiveMessage(message));
    UnitAssert(message.Questions().size() > 0);
    UnitAssert(message.Answers().size() > 0);
    for (auto & answer : message.Answers()) {
      UnitAssert(answer.Name() == "ietf.org");
      Dns::RRDataNSEC  *nsec = answer.Data<Dns::RRDataNSEC>();
      if (UnitAssert(nsec != nullptr)) {
        UnitAssert(! nsec->NextDomainName().empty());
        UnitAssert(! nsec->Bitmaps().empty());
      }
    }
  }
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
void TestTcpIETFRRSIG()
{
  Dns::Message  message;
  message.Header().RecursionDesired(true);
  Dns::MessageQuestion  question("ietf.org",
                                 Dns::MessageQuestion::k_typeRRSIG,
                                 Dns::MessageQuestion::k_classIN);
  message.Questions().push_back(question);
  message.EnableEDNS(4096, true);
  
  Dns::NameServer  nameServer(Ipv4Address("8.8.8.8"));
  if (UnitAssert(nameServer.WriteMessage(message))) {
    UnitAssert(nameServer.ReadMessage(message));
    UnitAssert(message.Questions().size() > 0);
    UnitAssert(message.Answers().size() > 0);
    for (auto & answer : message.Answers()) {
      UnitAssert(answer.Name() == "ietf.org");
      Dns::RRDataRRSIG  *rrsig = answer.Data<Dns::RRDataRRSIG>();
      if (UnitAssert(rrsig != nullptr)) {
        UnitAssert(! rrsig->SignersName().empty());
        UnitAssert(! rrsig->Signature().empty());
      }
    }
  }
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
void TestIETFRRSIG()
{
  Dns::Message  message;
  message.Header().RecursionDesired(true);
  Dns::MessageQuestion  question("ietf.org",
                                 Dns::MessageQuestion::k_typeRRSIG,
                                 Dns::MessageQuestion::k_classIN);
  message.Questions().push_back(question);
  message.EnableEDNS(4096, true);
  
  Dns::NameServer  nameServer(Ipv4Address("8.8.8.8"));
  if (UnitAssert(nameServer.SendMessage(message) > 0)) {
    UnitAssert(nameServer.ReceiveMessage(message));
    UnitAssert(message.Questions().size() > 0);
    UnitAssert(message.Answers().size() > 0);
    for (auto & answer : message.Answers()) {
      UnitAssert(answer.Name() == "ietf.org");
      Dns::RRDataRRSIG  *rrsig = answer.Data<Dns::RRDataRRSIG>();
      if (UnitAssert(rrsig != nullptr)) {
        UnitAssert(! rrsig->SignersName().empty());
        UnitAssert(! rrsig->Signature().empty());
      }
    }
  }
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
void TestIETFDS()
{
  Dns::Message  message;
  message.Header().RecursionDesired(true);
  Dns::MessageQuestion  question("ietf.org",
                                 Dns::MessageQuestion::k_typeDS,
                                 Dns::MessageQuestion::k_classIN);
  message.Questions().push_back(question);
  
  Dns::NameServer  nameServer(Ipv4Address("8.8.8.8"));
  if (UnitAssert(nameServer.SendMessage(message) > 0)) {
    UnitAssert(nameServer.ReceiveMessage(message));
    UnitAssert(message.Questions().size() > 0);
    UnitAssert(message.Answers().size() > 0);
    for (auto & answer : message.Answers()) {
      UnitAssert(answer.Name() == "ietf.org");
      Dns::RRDataDS  *ds = answer.Data<Dns::RRDataDS>();
      if (UnitAssert(ds != nullptr)) {
        UnitAssert(! ds->Digest().empty());
      }
    }
  }
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
void TestCAIDALOC()
{
  Dns::Message  message;
  message.Header().RecursionDesired(true);
  Dns::MessageQuestion  question("caida.org",
                                 Dns::MessageQuestion::k_typeLOC,
                                 Dns::MessageQuestion::k_classIN);
  message.Questions().push_back(question);
  
  Dns::NameServer  nameServer(Ipv4Address("8.8.8.8"));
  if (UnitAssert(nameServer.SendMessage(message) > 0)) {
    UnitAssert(nameServer.ReceiveMessage(message));
    UnitAssert(message.Questions().size() > 0);
    UnitAssert(message.Answers().size() > 0);
    for (auto & answer : message.Answers()) {
      UnitAssert(answer.Name() == "caida.org");
      Dns::RRDataLOC  *loc = answer.Data<Dns::RRDataLOC>();
      UnitAssert(loc != nullptr);
    }
  }
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
void TestCAIDATXT()
{
  Dns::Message  message;
  message.Header().RecursionDesired(true);
  Dns::MessageQuestion  question("caida.org",
                                 Dns::MessageQuestion::k_typeTXT,
                                 Dns::MessageQuestion::k_classIN);
  message.Questions().push_back(question);
  
  Dns::NameServer  nameServer(Ipv4Address("8.8.8.8"));
  if (UnitAssert(nameServer.SendMessage(message) > 0)) {
    UnitAssert(nameServer.ReceiveMessage(message));
    UnitAssert(message.Questions().size() > 0);
    UnitAssert(message.Answers().size() > 0);
    for (auto & answer : message.Answers()) {
      UnitAssert(answer.Name() == "caida.org");
      Dns::RRDataTXT  *txt = answer.Data<Dns::RRDataTXT>();
      if (UnitAssert(txt != nullptr)) {
        UnitAssert(! txt->Txt().empty());
      }
    }
  }
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
void TestMCPLEXMX()
{
  Dns::Message  message;
  message.Header().RecursionDesired(true);
  Dns::MessageQuestion  question("mcplex.net",
                                 Dns::MessageQuestion::k_typeMX,
                                 Dns::MessageQuestion::k_classIN);
  message.Questions().push_back(question);
  
  Dns::NameServer  nameServer(Ipv4Address("8.8.8.8"));
  if (UnitAssert(nameServer.SendMessage(message) > 0)) {
    UnitAssert(nameServer.ReceiveMessage(message));
    UnitAssert(message.Questions().size() > 0);
    UnitAssert(message.Answers().size() > 0);
    for (auto & answer : message.Answers()) {
      UnitAssert(answer.Name() == "mcplex.net");
      Dns::RRDataMX  *mx = answer.Data<Dns::RRDataMX>();
      if (UnitAssert(mx != nullptr)) {
        UnitAssert(! mx->Exchange().empty());
      }
    }
  }
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
void TestJUNCTIONSRV()
{
  Dns::Message  message;
  message.Header().RecursionDesired(true);
  Dns::MessageQuestion  question("_xmpp-client._tcp.junctionnetworks.com",
                                 Dns::MessageQuestion::k_typeSRV,
                                 Dns::MessageQuestion::k_classIN);
  message.Questions().push_back(question);
  
  Dns::NameServer  nameServer(Ipv4Address("8.8.8.8"));
  if (UnitAssert(nameServer.SendMessage(message) > 0)) {
    UnitAssert(nameServer.ReceiveMessage(message));
    UnitAssert(message.Questions().size() > 0);
    UnitAssert(message.Answers().size() > 0);
    for (auto & answer : message.Answers()) {
      UnitAssert(answer.Name() == "_xmpp-client._tcp.junctionnetworks.com");
      Dns::RRDataSRV  *srv = answer.Data<Dns::RRDataSRV>();
      if (UnitAssert(srv != nullptr)) {
        UnitAssert(srv->Port() != 0);
        UnitAssert(! srv->Target().empty());
      }
    }
  }
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
int main(int argc, char *argv[])
{
  TestGoogleA();
  TestGoogleAAAA();
  TestGoogleSOA();
  TestGooglePTR();
  TestTwitCNAME();
  TestGRCCAA();
  TestIETFDNSKEY();
  TestIETFNSEC();
  // TestIETFRRSIG();
  TestTcpIETFRRSIG();
  TestIETFDS();
  TestCAIDALOC();
  TestCAIDATXT();
  TestMCPLEXMX();
  TestJUNCTIONSRV();
  
  if (Assertions::Total().Failed()) {
    Assertions::Print(cerr, true);
    return 1;
  }
  else {
    cout << Assertions::Total() << " passed" << endl;
    return 0;
  }
}
