//===========================================================================
// @(#) $DwmPath: dwm/DwmDns/tags/DwmDns-0.2.6/classes/include/DwmDnsNameServer.hh 10735 $
// @(#) $Id: DwmDnsNameServer.hh 10735 2020-05-07 18:18:27Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2017, 2018, 2020
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmDnsNameServer.hh
//!  \brief Dwm::Dns::NameServer class definition
//---------------------------------------------------------------------------

#ifndef _DNSNAMESERVER_HH_
#define _DNSNAMESERVER_HH_

extern "C" {
  #include <sys/socket.h>
  #include <sys/select.h>
  #include <netinet/in.h>
  #include <unistd.h>
}

#include <chrono>
#include <string>
#include <tuple>
#include <vector>

#include "DwmIpv4Address.hh"
#include "DwmDnsMessage.hh"

namespace Dwm {

  namespace Dns {
    
    //------------------------------------------------------------------------
    //!  Encapsulates a DNS name server, from the perspective of a DNS client.
    //------------------------------------------------------------------------
    class NameServer
    {
    public:
      NameServer();
      
      //----------------------------------------------------------------------
      //!  Constructs a name server whose IPv4 address is @c addr.
      //----------------------------------------------------------------------
      NameServer(const struct in_addr & addr);

      //----------------------------------------------------------------------
      //!  
      //----------------------------------------------------------------------
      NameServer(const Ipv4Address & addr);
      
      //----------------------------------------------------------------------
      //!  Constructs a name server whose IPv6 address is @c addr.
      //----------------------------------------------------------------------
      NameServer(const struct in6_addr & addr);

      //----------------------------------------------------------------------
      //!  Constructs a name server whose IP address is @c addr.  @c addr
      //!  may be an IPv4 or IPv6 address.  Throws std::invalid_argument if
      //!  @c addr is not a valid IPv4 or IPv6 address.
      //----------------------------------------------------------------------
      NameServer(const std::string & addr);
      
      //----------------------------------------------------------------------
      //!  Copy constructor deleted because we don't want to share socket
      //!  descriptors.
      //----------------------------------------------------------------------
      NameServer(const NameServer &) = delete;
      
      //----------------------------------------------------------------------
      //!  Copy operator deleted because we don't want to share socket
      //!  descriptors.
      //----------------------------------------------------------------------
      NameServer & operator = (const NameServer &) = delete;

      //----------------------------------------------------------------------
      //!  Move constructor
      //----------------------------------------------------------------------
      NameServer(NameServer && ns) noexcept;

      //----------------------------------------------------------------------
      //!  Move operator
      //----------------------------------------------------------------------
      NameServer & operator = (NameServer && ns) noexcept;
      
      //----------------------------------------------------------------------
      //!  Destructor
      //----------------------------------------------------------------------
      ~NameServer();
      
      typedef std::tuple<sa_family_t,const struct sockaddr *,socklen_t> AddrInfo;
      
      //----------------------------------------------------------------------
      //!  Returns the address of the name server.
      //----------------------------------------------------------------------
      AddrInfo Address() const;
      
      //----------------------------------------------------------------------
      //!  Sends the given @c message to the name server via UDP.  Returns
      //!  true on success, false on failure.
      //----------------------------------------------------------------------
      bool SendMessage(const Message & message);
      
      //----------------------------------------------------------------------
      //!  Sends the given message to the name server via TCP.  Returns true
      //!  on success, false on failure.
      //----------------------------------------------------------------------
      bool WriteMessage(const Message & message);
      
      //----------------------------------------------------------------------
      //!  Receives a @c message from the name server via UDP.  Returns true
      //!  on success, false on failure.
      //----------------------------------------------------------------------
      bool ReceiveMessage(Message & message);
      
      //----------------------------------------------------------------------
      //!  Reads a @c message from the name server via TCP.  Returns true on
      //!  success, false on failure.
      //----------------------------------------------------------------------
      bool ReadMessage(Message & message, int32_t readTimeoutMsecs = 2000);

      //----------------------------------------------------------------------
      //!  Returns true if there is a UDP message ready to be read via
      //!  ReceiveMessage().  @c timeout is the time to wait for a message
      //!  to be ready.
      //----------------------------------------------------------------------
      bool AnswerWaiting(const struct timeval & timeout) const;
      
    private:
      sa_family_t      _addrFamily;
      union {
        struct sockaddr_in   in4Addr;
        struct sockaddr_in6  in6Addr;
      } _addr;
      int              _udpfd;
      int              _tcpfd;
      
      bool Open(bool tcp = false);
    };
    
  }  // namespace Dns

}  // namespace Dwm

#endif  // _DNSNAMESERVER_HH_
