//===========================================================================
// @(#) $DwmPath: dwm/DwmDns/tags/DwmDns-0.2.6/classes/include/DwmDnsRRDataDHCID.hh 10129 $
// @(#) $Id: DwmDnsRRDataDHCID.hh 10129 2018-01-27 04:20:26Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2018
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmDnsRRDataDHCID.hh
//!  \brief Dwm::Dns::RRDataDHCID class definition
//---------------------------------------------------------------------------

#ifndef _DWMDNSRRDATADHCID_HH_
#define _DWMDNSRRDATADHCID_HH_

#include <cstdint>
#include <iostream>
#include <string>

#include "DwmDnsRRData.hh"

namespace Dwm {

  namespace Dns {

    //------------------------------------------------------------------------
    //!  @ingroup rrdatagroup
    //!  Encapsulates DHCID resource record data.
    //------------------------------------------------------------------------
    class RRDataDHCID
      : public RRData
    {
    public:
      static const uint16_t  k_rrtype = 49;
      
      static const uint16_t  k_idDHCPv4HtypeHlen = 0;
      static const uint16_t  k_idDHCPv4ClientId  = 1;
      static const uint16_t  k_idDUID            = 2;

      static const uint8_t  k_digestTypeReserved = 0;
      static const uint8_t  k_digestTypeSHA256   = 1;
      
      //----------------------------------------------------------------------
      //!  Default constructor
      //----------------------------------------------------------------------
      RRDataDHCID();
      
      //----------------------------------------------------------------------
      //!  Construct with the given @c id, @c digestType and @c digest.
      //----------------------------------------------------------------------
      RRDataDHCID(uint16_t id, uint8_t digestType,
                  const std::string & digest);
      
      //----------------------------------------------------------------------
      //!  Equal-to operator, mostly for unit testing.
      //----------------------------------------------------------------------
      bool operator == (const RRDataDHCID & dhcid) const;
      
      //----------------------------------------------------------------------
      //!  Returns the ID in the DHCID resource record data.
      //----------------------------------------------------------------------
      uint16_t Id() const;
      
      //----------------------------------------------------------------------
      //!  Sets and returns the ID in the DHCID resource record data.
      //----------------------------------------------------------------------
      uint16_t Id(uint16_t id);
      
      //----------------------------------------------------------------------
      //!  Returns the digest type in the DHCID resource record data.
      //----------------------------------------------------------------------
      uint8_t DigestType() const;
      
      //----------------------------------------------------------------------
      //!  Sets and returns the digest type in the DHCID resource record data.
      //----------------------------------------------------------------------
      uint8_t DigestType(uint8_t digestType);
      
      //----------------------------------------------------------------------
      //!  Returns the digest in the DHCID resource record data.
      //----------------------------------------------------------------------
      const std::string & Digest() const;
      
      //----------------------------------------------------------------------
      //!  Sets and returns the digest in the DHCID resource record data.
      //----------------------------------------------------------------------
      const std::string & Digest(const std::string & digest);

      //----------------------------------------------------------------------
      //!  Encodes the DHCID resource record data into buffer @ pkt of length
      //!  @c pktlen, starting at @c ptr.  Returns the address immediately
      //!  following the encoded DHCID resource record data in @c pkt on
      //!  success.  Throws std::out_of_range if @c pkt is too short to
      //!  contain the encoded DHCID resource record data.
      //----------------------------------------------------------------------
      uint8_t *Encode(uint8_t *pkt, uint8_t *ptr, uint16_t pktlen,
                      LabelPositions & lps) const;

      //----------------------------------------------------------------------
      //!  Decodes the DHCID resource record data from buffer @c pkt of length
      //!  @c pktlen, starting at @c ptr.  Returns the address immediately
      //!  following the encoded  DHCID resource record data in @c pkt on
      //!  success.  Throws std::out_of_range if @c pkt was too short to
      //!  contain an encoded DHCID resource record data.
      //----------------------------------------------------------------------
      const uint8_t *Decode(const uint8_t *pkt, const uint8_t *ptr,
                            uint16_t pktlen, uint16_t rdlen);

      //----------------------------------------------------------------------
      //!  Print the DHCID resource record data to an ostream in human
      //!  readable form.
      //----------------------------------------------------------------------
      friend std::ostream & operator << (std::ostream & os,
                                         const RRDataDHCID & dhcid);
      
    private:
      uint16_t     _id;
      uint8_t      _digestType;
      std::string  _digest;
    };
    
  }  // namespace Dns

}  // namespace Dwm

#endif  // _DWMDNSRRDATADHCID_HH_
