//===========================================================================
// @(#) $DwmPath: dwm/DwmDns/tags/DwmDns-0.2.6/classes/include/DwmDnsResourceRecord.hh 10136 $
// @(#) $Id: DwmDnsResourceRecord.hh 10136 2018-01-28 06:01:09Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2018
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmDnsResourceRecord.hh
//!  \brief Dwm::Dns::ResourceRecord class definition
//---------------------------------------------------------------------------

#ifndef _DWMDNSRESOURCERECORD_HH_
#define _DWMDNSRESOURCERECORD_HH_

#include "DwmDnsPortability.hh"

#include <cstdint>
#include <iostream>
#include <map>
#include <string>
#include <typeinfo>
#ifdef HAVE_VARIANT
#include <variant>
#define MY_VARIANT std::variant
#define MY_GET_IF std::get_if
#else
#include <mpark_variant>
#define MY_VARIANT mpark::variant
#define MY_GET_IF mpark::get_if  
#endif

#include "DwmDnsLabelPositions.hh"
#include "DwmDnsRRDataA.hh"
#include "DwmDnsRRDataNS.hh"
#include "DwmDnsRRDataCNAME.hh"
#include "DwmDnsRRDataAAAA.hh"
#include "DwmDnsRRDataSOA.hh"
#include "DwmDnsRRDataMB.hh"
#include "DwmDnsRRDataMG.hh"
#include "DwmDnsRRDataMR.hh"
#include "DwmDnsRRDataPTR.hh"
#include "DwmDnsRRDataHINFO.hh"
#include "DwmDnsRRDataMINFO.hh"
#include "DwmDnsRRDataMX.hh"
#include "DwmDnsRRDataTXT.hh"
#include "DwmDnsRRDataRP.hh"
#include "DwmDnsRRDataKEY.hh"
#include "DwmDnsRRDataLOC.hh"
#include "DwmDnsRRDataSRV.hh"
#include "DwmDnsRRDataCERT.hh"
#include "DwmDnsRRDataDS.hh"
#include "DwmDnsRRDataSSHFP.hh"
#include "DwmDnsRRDataRRSIG.hh"
#include "DwmDnsRRDataNSEC.hh"
#include "DwmDnsRRDataDNSKEY.hh"
#include "DwmDnsRRDataDHCID.hh"
#include "DwmDnsRRDataNSEC3.hh"
#include "DwmDnsRRDataNSEC3PARAM.hh"
#include "DwmDnsRRDataTLSA.hh"
#include "DwmDnsRRDataSMIMEA.hh"
#include "DwmDnsRRDataOPENPGPKEY.hh"
#include "DwmDnsRRDataURI.hh"
#include "DwmDnsRRDataCAA.hh"
#include "DwmDnsRRDataUnknown.hh"

namespace Dwm {

  namespace Dns {

    //------------------------------------------------------------------------
    //!  @ingroup messagegroup
    //!  Encapsulates a DNS resource record.  See RFC1035 for details.
    //------------------------------------------------------------------------
    class ResourceRecord
    {
    public:
      //----------------------------------------------------------------------
      //!  Resource record types
      //----------------------------------------------------------------------
      typedef enum {
        k_typeA           = 1,      //! host address
        k_typeNS          = 2,      //! authoritative nameserver
        k_typeMD          = 3,      //! mail destination (old, use MX)
        k_typeMF          = 4,      //! mail forwarder (obsolete, use MX)
        k_typeCNAME       = 5,      //! canonical name for an alias
        k_typeSOA         = 6,      //! marks the start of a zone of authority
        k_typeMB          = 7,      //! mailbox domain name (EXPERIMENTAL)
        k_typeMG          = 8,      //! mail group member (EXPERIMENTAL)
        k_typeMR          = 9,      //! mail rename domain name (EXPERIMENTAL)
        k_typeNULL        = 10,     //! null RR (EXPERIMENTAL)
        k_typeWKS         = 11,     //! well known service description
        k_typePTR         = 12,     //! domain name pointer
        k_typeHINFO       = 13,     //! host information
        k_typeMINFO       = 14,     //! mailbox or mail list information
        k_typeMX          = 15,     //! mail exchange
        k_typeTXT         = 16,     //! text strings
        k_typeRP          = 17,     //! Responsible Person (RFC1183)
        k_typeKEY         = 25,     //! public key (RFC2535)
        k_typeAAAA        = 28,     //! host IPv6 address (RFC3596)
        k_typeLOC         = 29,     //! location (LOC) (RFC1876)
        k_typeSRV         = 33,     //! server selection (RFC2782)
        k_typeCERT        = 37,     //! certificate (RFC4398)
        k_typeOPT         = 41,     //! OPT (for EDNS, RFC6891)
        k_typeDS          = 43,     //! DS (RFC4034)
        k_typeSSHFP       = 44,     //! SSH fingerprint (RFC4255)
        k_typeRRSIG       = 46,     //! RRSIG (RFC4034)
        k_typeNSEC        = 47,     //! NSEC (RFC4034)
        k_typeDNSKEY      = 48,     //! DNSKEY (RFC4034)
        k_typeDHCID       = 49,     //! DHCID (RFC4701)
        k_typeNSEC3       = 50,     //! NSEC3 (RFC5155)
        k_typeNSEC3PARAM  = 51,     //! NSEC3PARAM (RFC5155)
        k_typeTLSA        = 52,     //! TLSA (RFC6698)
        k_typeSMIMEA      = 53,     //! SMIMEA (RFC8162)
        k_typeCDS         = 59,     //! CDS (RFC7344)
        k_typeCDNSKEY     = 60,     //! CDNSKEY (RFC7344)
        k_typeOPENPGPKEY  = 61,     //! OPENPGPKEY (RFC7929)
        k_typeSPF         = 99,     //! RFC4408 but obsoleted in RFC7208
        k_typeURI         = 256,    //! URI (RFC7553)
        k_typeCAA         = 257,    //! cert. authority auth. (RFC6844)
        k_typeUNKNOWN     = 0xFFFF  //! not a real TYPE
      } TypeEnum;

      //----------------------------------------------------------------------
      //!  Resource record classes
      //----------------------------------------------------------------------
      typedef enum {
        k_classIN       = 1,       //! Internet
        k_classCH       = 3,       //! CHAOS
        k_classHS       = 4,       //! Hesiod
        k_classUNKNOWN  = 0xFFFF   //! not a real CLASS
      } ClassEnum;
      
      //----------------------------------------------------------------------
      //!  Default constructor
      //----------------------------------------------------------------------
      ResourceRecord();
      
      //----------------------------------------------------------------------
      //!  Copy constructor
      //----------------------------------------------------------------------
      ResourceRecord(const ResourceRecord & rr);
      
      //----------------------------------------------------------------------
      //!  operator =
      //----------------------------------------------------------------------
      ResourceRecord & operator = (const ResourceRecord & rr);
      
      //----------------------------------------------------------------------
      //!  Destructor
      //----------------------------------------------------------------------
      ~ResourceRecord() = default;
      
      //----------------------------------------------------------------------
      //!  Equal-to operator, mostly for unit testing.
      //----------------------------------------------------------------------
      bool operator == (const ResourceRecord & rr) const;
      
      //----------------------------------------------------------------------
      //!  Returns the contained domain name.
      //----------------------------------------------------------------------
      const std::string & Name() const;
      
      //----------------------------------------------------------------------
      //!  Sets and returns the contained domain name.
      //----------------------------------------------------------------------
      const std::string & Name(const std::string & name);
      
      //----------------------------------------------------------------------
      //!  Returns the contained type.
      //----------------------------------------------------------------------
      TypeEnum Type() const;
      
      //----------------------------------------------------------------------
      //!  Sets and returns the contained type.
      //----------------------------------------------------------------------
      TypeEnum Type(TypeEnum typeEnum);
      
      //----------------------------------------------------------------------
      //!  Returns the contained class.
      //----------------------------------------------------------------------
      uint16_t Class() const;
      
      //----------------------------------------------------------------------
      //!  Sets and returns the contained class.
      //----------------------------------------------------------------------
      uint16_t Class(uint16_t cl);
      
      //----------------------------------------------------------------------
      //!  Returns the TTL.
      //----------------------------------------------------------------------
      uint32_t TTL() const;
      
      //----------------------------------------------------------------------
      //!  Sets and returns the TTL.
      //----------------------------------------------------------------------
      uint32_t TTL(uint32_t ttl);

      //----------------------------------------------------------------------
      //!  Encodes the resource record into buffer @ pkt of length @c pktlen,
      //!  starting at @c ptr.  Returns the address immediately following the
      //!  encoded resource record in @c pkt on success.  Throws
      //!  std::out_of_range if @c pkt is too short to contain the encoded
      //!  resource record.
      //----------------------------------------------------------------------
      uint8_t *Encode(uint8_t *pkt, uint8_t *ptr, uint16_t pktlen,
                      LabelPositions & lps) const;
      
      //----------------------------------------------------------------------
      //!  Decodes the resource record from buffer @c pkt of length @c pktlen,
      //!  starting at @c ptr.  Returns the address immediately following the
      //!  encoded  DHCID resource record data in @c pkt on success.  Throws
      //!  std::out_of_range if @c pkt was too short to contain an encoded
      //!  resource record.
      //----------------------------------------------------------------------
      const uint8_t *Decode(const uint8_t *pkt, const uint8_t *ptr,
                            uint16_t pktlen);
      
      //----------------------------------------------------------------------
      //!  Returns a pointer to const of the contained RRDATA.
      //----------------------------------------------------------------------
      template <typename T>
      const T * Data() const
      {
        return MY_GET_IF<T>(&_rrdata);
      }

      //----------------------------------------------------------------------
      //!  Sets and returns the RRDATA of the resource record.  Note that
      //!  this should normally only be called with one or two arguments,
      //!  which means we're using a known RRDATA encoding.  If you specify
      //!  the third argument (resource record type), it MUST be a type
      //!  that we don't handle and @c rrdata must be RRDataUnknown (which
      //!  is just a blob stored in a strong).
      //----------------------------------------------------------------------
      template <typename T>
      const T *Data(const T & rrdata, uint16_t rrClass = k_classIN,
                    uint16_t rrType = T::k_rrtype)
      {
        _class = rrClass;
        _type = (TypeEnum)rrType;
        _rrdata = rrdata;
        return MY_GET_IF<T>(&_rrdata);
      }

      //----------------------------------------------------------------------
      //!  Returns a pointer to mutable contained RRDATA.
      //----------------------------------------------------------------------
      template <typename T>
      T * Data()
      {
        return MY_GET_IF<T>(&_rrdata);
      }
      
      //----------------------------------------------------------------------
      //!  Prints the resource record to an ostream in human readable form.
      //----------------------------------------------------------------------
      friend std::ostream & operator << (std::ostream & os,
                                         const ResourceRecord & rr);

      //----------------------------------------------------------------------
      //!  Returns the mnemonic for the given resource record type @ rrtype.
      //----------------------------------------------------------------------
      static std::string TypeName(uint16_t rrtype);
      
      //----------------------------------------------------------------------
      //!  Returns the mnemonic for the given recource record class @ rrclass.
      //----------------------------------------------------------------------
      static std::string ClassName(uint16_t rrclass);
      
    protected:
      std::string  _name;
      TypeEnum     _type;
      uint16_t     _class;
      uint32_t     _ttl;
      MY_VARIANT<RRDataA,             // 0
                 RRDataNS,            // 1
                 RRDataCNAME,	        // 2
                 RRDataSOA,           // 3
                 RRDataMB,            // 4
                 RRDataMG,            // 5
                 RRDataMR,            // 6
                 RRDataPTR,           // 7
                 RRDataHINFO,         // 8
                 RRDataMINFO,         // 9
                 RRDataMX,            // 10
                 RRDataTXT,           // 11
                 RRDataRP,            // 12
                 RRDataKEY,           // 13
                 RRDataAAAA,          // 14
                 RRDataLOC,           // 15
                 RRDataSRV,           // 16
                 RRDataCERT,          // 17
                 RRDataDS,            // 18
                 RRDataSSHFP,         // 19
                 RRDataRRSIG,         // 20
                 RRDataNSEC,          // 21
                 RRDataDNSKEY,        // 22
                 RRDataDHCID,         // 23
                 RRDataNSEC3,         // 24
                 RRDataNSEC3PARAM,    // 25
                 RRDataTLSA,          // 26
                 RRDataSMIMEA,        // 27
                 RRDataOPENPGPKEY,    // 28
                 RRDataURI,           // 29
                 RRDataCAA,           // 30
                 RRDataUnknown        // 31
                 > _rrdata;

      static std::map<uint16_t,std::string>  _classNames;
      static std::map<uint16_t,std::string>  _typeNames;

#if 0
      //  xxx : no longer used, but could be revived in the future if
      //  I need to be able to fetch contained RRDATA by resource record
      //  type enumeration.
      static constexpr std::pair<uint16_t,size_t>  k_rrTypeArr[] = {
        { k_typeA,          0  },
        { k_typeNS,         1  },
        { k_typeCNAME,      2  },
        { k_typeSOA,        3  },
        { k_typeMB,         4  },
        { k_typeMG,         5  },
        { k_typeMR,         6  },
        { k_typePTR,        7  },
        { k_typeHINFO,      8  },
        { k_typeMINFO,      9  },
        { k_typeMX,         10 },
        { k_typeTXT,        11 },
        { k_typeRP,         12 },
        { k_typeKEY,        13 },
        { k_typeAAAA,       14 },
        { k_typeLOC,        15 },
        { k_typeSRV,        16 },
        { k_typeCERT,       17 },
        { k_typeDS,         18 },
        { k_typeSSHFP,      19 },
        { k_typeRRSIG,      20 },
        { k_typeNSEC,       21 },
        { k_typeDNSKEY,     22 },
        { k_typeDHCID,      23 },
        { k_typeNSEC3,      24 },
        { k_typeNSEC3PARAM, 25 },
        { k_typeTLSA,       26 },
        { k_typeSMIMEA,     27 },
        { k_typeCDS,        18 },
        { k_typeCDNSKEY,    22 },
        { k_typeOPENPGPKEY, 28 },
        { k_typeURI,        29 },
        { k_typeCAA,        30 }
      };

      //----------------------------------------------------------------------
      //!  
      //----------------------------------------------------------------------
      static constexpr auto k_rrTypeArrSize =
        sizeof k_rrTypeArr / sizeof k_rrTypeArr[0];

      //----------------------------------------------------------------------
      //!  
      //----------------------------------------------------------------------
      static constexpr size_t RRDataIndex(uint16_t rrtype,
                                          int range = k_rrTypeArrSize)
      {
        return (k_rrTypeArr[range - 1].first == rrtype) ?
          k_rrTypeArr[range - 1].second
          : RRDataIndex(rrtype, range - 1);
      }
#endif

      std::ostream & PrintData(std::ostream & os) const;
    };

    
  }  // namespace Dns

}  // namespace Dwm

#endif  // _DWMDNSRESOURCERECORD_HH_
