//===========================================================================
// @(#) $DwmPath: dwm/DwmDns/tags/DwmDns-0.2.6/classes/src/DwmDnsUtils.cc 10124 $
// @(#) $Id: DwmDnsUtils.cc 10124 2018-01-27 02:06:42Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2018
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmDnsUtils.cc
//!  \brief NOT YET DOCUMENTED
//---------------------------------------------------------------------------

#include <cstdio>
#include <cstring>
#include <sstream>
#include <stdexcept>

#include "DwmSvnTag.hh"
#include "DwmDnsUtils.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/DwmDns/tags/DwmDns-0.2.6/classes/src/DwmDnsUtils.cc 10124 $");

using namespace std;

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
bool operator == (const in6_addr & addr1, const in6_addr & addr2)
{
  return (memcmp(&addr1, &addr2, sizeof(in6_addr)) == 0);
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
bool operator < (const in6_addr & addr1, const in6_addr & addr2)
{
  return (memcmp(&addr1, &addr2, sizeof(in6_addr)) < 0);
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
bool operator == (const in_addr & addr1, const in_addr & addr2)
{
  return (addr1.s_addr == addr2.s_addr);
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
bool operator < (const in_addr & addr1, const in_addr & addr2)
{
  return (ntohl(addr1.s_addr) < ntohl(addr2.s_addr));
}

namespace Dwm {

  namespace Dns {

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint8_t *EncodeCharacterString(const string & s, uint8_t *pkt,
                                   uint8_t *ptr, uint16_t pktlen)
    {
      if (((ptr + 1 + s.size()) <= (ptr + pktlen))
          && (s.size() < 256)) {
        *ptr++ = s.size() & 0xFF;
        memcpy(ptr, s.c_str(), s.size());
        ptr += s.size();
      }
      else {
        throw out_of_range("character string will not fit in packet");
      }
      return ptr;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const uint8_t *DecodeCharacterString(string & s, const uint8_t *pkt,
                                         const uint8_t *ptr, uint16_t pktlen)
    {
      s.clear();
      if (ptr < (pkt + pktlen)) {
        uint8_t  len = *ptr++;
        if ((ptr + len) <= (pkt + pktlen)) {
          s.assign((const char *)ptr, len);
          ptr += len;
        }
        else {
          throw out_of_range("packet too short for character string");
        }
      }
      else {
        throw out_of_range("packet too short for character string");
      }
      return ptr;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void ToArpa(const in_addr & inAddr, std::string & arpa)
    {
      in_addr_t      in4Addr = ntohl(inAddr.s_addr);
      ostringstream  os;
      os << (uint16_t)(in4Addr & 0xFF) << '.'
         << (uint16_t)((in4Addr >> 8) & 0xFF) << '.'
         << (uint16_t)((in4Addr >> 16) & 0xFF) << '.'
         << (uint16_t)(in4Addr >> 24) << ".in-addr.arpa";
      arpa = os.str();
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void ToArpa(const in6_addr & in6Addr, std::string & arpa)
    {
      const uint8_t  *ps = (const uint8_t *)&in6Addr.s6_addr;
      const uint8_t  *pe = ps + (sizeof(in6Addr.s6_addr) - 1);
      ostringstream   os;
      os << hex;
      while (pe > ps) {
        os << (uint16_t)((*pe) & 0x0f) << '.'
           << (uint16_t)((*pe) >> 4) << '.';
        --pe;
      }
      os << (uint16_t)((*pe) & 0x0f) << '.' << (*pe >> 4) << ".ip6.arpa";
      arpa = os.str();
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool ToArpa(const string & ipAddr, string & arpa)
    {
      arpa.clear();
      if (! ipAddr.empty()) {
        in6_addr  in6Addr;
        in_addr   inAddr;
        if (inet_pton(AF_INET6, ipAddr.c_str(), &in6Addr) == 1) {
          ToArpa(in6Addr, arpa);
        }
        else if (inet_pton(AF_INET, ipAddr.c_str(), &inAddr) == 1) {
          ToArpa(inAddr, arpa);
        }
      }
      return (! arpa.empty());
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool FromArpa(const string & arpa, string & ipAddr)
    {
      ipAddr.clear();
      uint8_t   nibs[32];
      if (sscanf(arpa.c_str(),
                 "%hhx.%hhx.%hhx.%hhx.%hhx.%hhx.%hhx.%hhx."
                 "%hhx.%hhx.%hhx.%hhx.%hhx.%hhx.%hhx.%hhx."
                 "%hhx.%hhx.%hhx.%hhx.%hhx.%hhx.%hhx.%hhx."
                 "%hhx.%hhx.%hhx.%hhx.%hhx.%hhx.%hhx.%hhx.ip6.arpa",
                 &nibs[0], &nibs[1], &nibs[2], &nibs[3],
                 &nibs[4], &nibs[5], &nibs[6], &nibs[7],
                 &nibs[8], &nibs[9], &nibs[10], &nibs[11],
                 &nibs[12], &nibs[13], &nibs[14], &nibs[15],
                 &nibs[16], &nibs[17], &nibs[18], &nibs[19],
                 &nibs[20], &nibs[21], &nibs[22], &nibs[23],
                 &nibs[24], &nibs[25], &nibs[26], &nibs[27],
                 &nibs[28], &nibs[29], &nibs[30], &nibs[31]) == 32) {
        in6_addr  in6Addr;
        uint8_t  *ps = (uint8_t *)&in6Addr.s6_addr;
        for (int8_t i = 31; i >= 0; i -= 2) {
          *ps++ = (nibs[i] << 4) | nibs[i - 1];
        }
        char  buf[INET6_ADDRSTRLEN] = { 0 };
        if (inet_ntop(AF_INET6, &in6Addr, buf, INET6_ADDRSTRLEN) == buf) {
          ipAddr = buf;
        }
      }
      else if (sscanf(arpa.c_str(), "%hhu.%hhu.%hhu.%hhu.in-addr.arpa",
                      &nibs[0], &nibs[1], &nibs[2], &nibs[3]) == 4) {
        in_addr  inAddr;
        inAddr.s_addr = ((uint32_t)nibs[3]) << 24;
        inAddr.s_addr |= ((uint32_t)nibs[2]) << 16;
        inAddr.s_addr |= ((uint32_t)nibs[1]) << 8;
        inAddr.s_addr |= (uint32_t)nibs[0];
        inAddr.s_addr = htonl(inAddr.s_addr);
        char  buf[INET_ADDRSTRLEN] = { 0 };
        if (inet_ntop(AF_INET, &inAddr, buf, INET_ADDRSTRLEN) == buf) {
          ipAddr = buf;
        }
      }
      return (! ipAddr.empty());
    }
    
  }  // namespace Dns

}  // namespace Dwm
