//===========================================================================
// @(#) $Name:$
// @(#) $Id: TestDnsResolver.cc 10069 2018-01-23 05:31:30Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2018
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file TestDnsResolver.cc
//!  \brief NOT YET DOCUMENTED
//---------------------------------------------------------------------------

#include "DwmSvnTag.hh"
#include "DwmUnitAssert.hh"
#include "DwmDnsResolver.hh"
#include "DwmDnsUtils.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/DwmDns/tags/DwmDns-0.2.6/classes/tests/TestDnsResolver.cc 10069 $");

using namespace std;
using namespace Dwm;

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
void TestGoogleA(Dns::Resolver & resolver)
{
  vector<Dns::RRDataA>  as;
  UnitAssert(resolver.Get("www.google.com", as));
  UnitAssert(! as.empty());
  for (auto a : as) {
    UnitAssert(a.InAddr().s_addr != INADDR_NONE);
  }
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
void TestGoogleAAAA(Dns::Resolver & resolver)
{
  vector<Dns::RRDataA>  aaaa;
  UnitAssert(resolver.Get("www.google.com", aaaa));
  UnitAssert(! aaaa.empty());
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
void TestGoogleSOA(Dns::Resolver & resolver)
{
  vector<Dns::RRDataSOA>  soa;
  UnitAssert(resolver.Get("google.com", soa));
  UnitAssert(! soa.empty());
  for (auto so : soa) {
    UnitAssert(! so.Mname().empty());
    UnitAssert(! so.Rname().empty());
  }
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
void TestGooglePTR(Dns::Resolver & resolver)
{
  vector<Dns::RRDataPTR>  ptr;
  UnitAssert(resolver.Get("8.8.8.8.in-addr.arpa", ptr));
  UnitAssert(! ptr.empty());
  for (auto pt : ptr) {
    UnitAssert(! pt.Ptr().empty());
  }
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
void TestTwitCNAME(Dns::Resolver & resolver)
{
  vector<Dns::RRDataCNAME>  cname;
  UnitAssert(resolver.Get("www.twit.tv", cname));
  UnitAssert(! cname.empty());
  for (auto cn : cname) {
    UnitAssert(! cn.Cname().empty());
  }
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
void TestGRCCAA(Dns::Resolver & resolver)
{
  vector<Dns::RRDataCAA>  caas;
  UnitAssert(resolver.Get("grc.com", caas));
  UnitAssert(! caas.empty());
  for (auto caa : caas) {
    UnitAssert(caa.Tag() == "issue");
    UnitAssert(caa.Value() == "digicert.com");
  }
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
void TestIETFDNSKEY(Dns::Resolver & resolver)
{
  vector<Dns::RRDataDNSKEY>  keys;
  UnitAssert(resolver.Get("ietf.org", keys, true));
  UnitAssert(! keys.empty());
  for (auto key : keys) {
    UnitAssert(! key.PublicKey().empty());
  }
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
void TestIETFNSEC(Dns::Resolver & resolver)
{
  vector<Dns::RRDataNSEC>  nsecs;
  UnitAssert(resolver.Get("ietf.org",nsecs));
  UnitAssert(! nsecs.empty());
  for (auto nsec : nsecs) {
    UnitAssert(! nsec.NextDomainName().empty());
    UnitAssert(! nsec.Bitmaps().empty());
  }
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
void TestIETFRRSIG(Dns::Resolver & resolver)
{
  vector<Dns::RRDataRRSIG>  rrsig;
  UnitAssert(resolver.Get("ietf.org", rrsig, true));
  UnitAssert(! rrsig.empty());
  for (auto rrs : rrsig) {
    UnitAssert(! rrs.SignersName().empty());
    UnitAssert(! rrs.Signature().empty());
  }
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
void TestIETFDS(Dns::Resolver & resolver)
{
  vector<Dns::RRDataDS>  dss;
  UnitAssert(resolver.Get("ietf.org", dss));
  UnitAssert(! dss.empty());
  for (auto ds : dss) {
    UnitAssert(! ds.Digest().empty());
  }
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
void TestCAIDALOC(Dns::Resolver & resolver)
{
  vector<Dns::RRDataLOC>  locs;
  UnitAssert(resolver.Get("caida.org", locs));
  UnitAssert(! locs.empty());
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
void TestCAIDATXT(Dns::Resolver & resolver)
{
  vector<Dns::RRDataTXT>  txts;
  UnitAssert(resolver.Get("caida.org", txts));
  UnitAssert(! txts.empty());
  for (auto txt : txts) {
    UnitAssert(! txt.Txt().empty());
    for (auto s : txt.Txt()) {
      UnitAssert(! s.empty());
    }
  }
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
void TestMcplexMX(Dns::Resolver & resolver)
{
  vector<Dns::RRDataMX>  mxs;
  UnitAssert(resolver.Get("mcplex.net", mxs));
  UnitAssert(! mxs.empty());
  for (auto mx : mxs) {
    UnitAssert(! mx.Exchange().empty());
  }
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
void TestGetHostByName(Dns::Resolver & resolver)
{
  vector<in6_addr>  in6Addrs;
  vector<in_addr>   inAddrs;
  UnitAssert(resolver.GetHostByName("localhost.", in6Addrs, inAddrs));
  if (UnitAssert(! in6Addrs.empty())) {
    in6_addr  in6Addr;
    inet_pton(AF_INET6, "::1", &in6Addr);
    UnitAssert(in6Addrs[0] == in6Addr);
  }
  if (UnitAssert(! inAddrs.empty())) {
    in_addr  inAddr;
    inet_pton(AF_INET, "127.0.0.1", &inAddr);
    UnitAssert(inAddrs[0] == inAddr);
  }
  return;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
void TestGetHostByAddr(Dns::Resolver & resolver)
{
  in6_addr        in6Addr;
  vector<string>  names;
  inet_pton(AF_INET6, "::1", &in6Addr);
  UnitAssert(resolver.GetHostByAddr(in6Addr, names));
  if (UnitAssert(! names.empty())) {
    UnitAssert(names[0] == "localhost");
  }

  in_addr  inAddr;
  inet_pton(AF_INET, "127.0.0.1", &inAddr);
  UnitAssert(resolver.GetHostByAddr(inAddr, names));
  if (UnitAssert(! names.empty())) {
    UnitAssert(names[0] == "localhost");
  }
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
int main(int argc, char *argv[])
{
  Dns::Resolver  resolver("./resolv.conf");
  TestGoogleA(resolver);
  TestGoogleAAAA(resolver);
  TestGoogleSOA(resolver);
  TestGooglePTR(resolver);
  TestTwitCNAME(resolver);
  TestGRCCAA(resolver);
  TestIETFDNSKEY(resolver);
  TestIETFNSEC(resolver);
  TestIETFRRSIG(resolver);
  TestIETFDS(resolver);
  TestCAIDALOC(resolver);
  TestCAIDATXT(resolver);
  TestMcplexMX(resolver);

  TestGetHostByName(resolver);
  TestGetHostByAddr(resolver);
  
  if (Assertions::Total().Failed()) {
    Assertions::Print(cerr, true);
    return 1;
  }
  else {
    cout << Assertions::Total() << " passed" << endl;
    return 0;
  }
}
