//===========================================================================
// @(#) $DwmPath: dwm/DwmDns/tags/DwmDns-0.2.7/classes/include/DwmDnsMessageQuestion.hh 11433 $
// @(#) $Id: DwmDnsMessageQuestion.hh 11433 2020-12-13 06:35:55Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2000, 2016, 2018
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmDnsMessageQuestion.hh
//!  \brief Dwm::Dns::MessageQuestion class definition
//---------------------------------------------------------------------------

#ifndef _DWMDNSMESSAGEQUESTION_HH_
#define _DWMDNSMESSAGEQUESTION_HH_

#include <cstdint>
#include <string>

#include "DwmDnsLabelPositions.hh"

namespace Dwm {

  namespace Dns {

    //------------------------------------------------------------------------
    //!  @ingroup messagegroup
    //!  Encapsulates a DNS message question.  See section 4.1.2 of RFC1035
    //!  for details.
    //------------------------------------------------------------------------
    class MessageQuestion
    {
    public:
      //----------------------------------------------------------------------
      //!  Query types.
      //----------------------------------------------------------------------
      static constexpr uint16_t  k_typeA            = 1;
      static constexpr uint16_t  k_typeNS           = 2;
      static constexpr uint16_t  k_typeCNAME        = 5;
      static constexpr uint16_t  k_typeSOA          = 6;
      static constexpr uint16_t  k_typeMB           = 7;
      static constexpr uint16_t  k_typeMG           = 8;
      static constexpr uint16_t  k_typeMR           = 9;
      static constexpr uint16_t  k_typePTR          = 12;
      static constexpr uint16_t  k_typeHINFO        = 13;
      static constexpr uint16_t  k_typeMINFO        = 14;
      static constexpr uint16_t  k_typeMX           = 15;
      static constexpr uint16_t  k_typeTXT          = 16;
      static constexpr uint16_t  k_typeRP           = 17;
      static constexpr uint16_t  k_typeKEY          = 25;
      static constexpr uint16_t  k_typeAAAA         = 28;
      static constexpr uint16_t  k_typeLOC          = 29;
      static constexpr uint16_t  k_typeSRV          = 33;
      static constexpr uint16_t  k_typeCERT         = 37;
      static constexpr uint16_t  k_typeDS           = 43;
      static constexpr uint16_t  k_typeSSHFP        = 44;
      static constexpr uint16_t  k_typeRRSIG        = 46;
      static constexpr uint16_t  k_typeNSEC         = 47;
      static constexpr uint16_t  k_typeDNSKEY       = 48;
      static constexpr uint16_t  k_typeDHCID        = 49;
      static constexpr uint16_t  k_typeNSEC3        = 50;
      static constexpr uint16_t  k_typeNSEC3PARAM   = 51;
      static constexpr uint16_t  k_typeTLSA         = 52;
      static constexpr uint16_t  k_typeSMIMEA       = 53;
      static constexpr uint16_t  k_typeCDS          = 59;
      static constexpr uint16_t  k_typeCDNSKEY      = 60;
      static constexpr uint16_t  k_typeOPENPGPKEY   = 61;
      static constexpr uint16_t  k_typeURI          = 256;
      static constexpr uint16_t  k_typeCAA          = 257;

      //----------------------------------------------------------------------
      //!  Query classes.
      //----------------------------------------------------------------------
      static constexpr uint16_t  k_classIN      = 1;
      static constexpr uint16_t  k_classCH      = 3;
      static constexpr uint16_t  k_classHS      = 4;
      static constexpr uint16_t  k_classANY     = 255;
      static constexpr uint16_t  k_classUNKNOWN = 0xFFFF;
      
      //----------------------------------------------------------------------
      //!  Constructs an empty question.
      //----------------------------------------------------------------------
      MessageQuestion();
      
      //----------------------------------------------------------------------
      //!  Constructs a question for domain name @c qname of type @c qtype
      //!  and class @c qclass.
      //----------------------------------------------------------------------
      MessageQuestion(const std::string & qname, uint16_t qtype,
                      uint16_t qclass);
      
      //----------------------------------------------------------------------
      //!  Equal-to operator, mostly useful for unit testing.
      //----------------------------------------------------------------------
      bool operator == (const MessageQuestion & question) const;
      
      //----------------------------------------------------------------------
      //!  Returns the QNAME from the question.
      //----------------------------------------------------------------------
      const std::string & QName() const;
      
      //----------------------------------------------------------------------
      //!  Sets and returns the QNAME of the question.
      //----------------------------------------------------------------------
      const std::string & QName(const std::string & qname);

      //----------------------------------------------------------------------
      //!  Returns the QTYPE of the question.
      //----------------------------------------------------------------------
      uint16_t QType() const;
      
      //----------------------------------------------------------------------
      //!  Sets and returns the QTYPE of the question.
      //----------------------------------------------------------------------
      uint16_t QType(uint16_t qtype);

      //----------------------------------------------------------------------
      //!  Returns a string representation of the QTYPE of the question.
      //----------------------------------------------------------------------
      std::string QTypeString() const;
      
      //----------------------------------------------------------------------
      //!  Returns the QCLASS of the question.
      //----------------------------------------------------------------------
      uint16_t QClass() const;
      
      //----------------------------------------------------------------------
      //!  Sets and returns the QCLASS of the question.
      //----------------------------------------------------------------------
      uint16_t QClass(uint16_t qclass);
      
      //----------------------------------------------------------------------
      //!  Encodes the question into buffer @ pkt of length @c pktlen,
      //!  starting at @c ptr.  Returns the address immediately following
      //!  the encoded question in @c pkt on success.  Throws
      //!  std::out_of_range if @c pkt is too short to contain the encoded
      //!  question.
      //----------------------------------------------------------------------
      uint8_t *Encode(uint8_t *pkt, uint8_t *ptr, uint16_t pktlen,
                      LabelPositions & lps) const;

      //----------------------------------------------------------------------
      //!  Decodes the question from buffer @c pkt of length @c pktlen,
      //!  starting at @c ptr.  Returns the address immediately following
      //!  the encoded question in @c pkt on success.  Throws
      //!  std::out_of_range if @c pkt was too short to contain an encoded
      //!  question.
      //----------------------------------------------------------------------
      const uint8_t *Decode(const uint8_t *pkt, const uint8_t *ptr,
                            uint16_t pktlen);
      
    private:
      std::string  _qname;
      uint16_t     _qtype;
      uint16_t     _qclass;

      static const std::map<uint16_t,std::string>  _qtypeNames;
    };
    
  }  // namespace Dns

}  // namespace Dwm

#endif  // _DWMDNSMESSAGEQUESTION_HH_
