//===========================================================================
// @(#) $DwmPath: dwm/DwmDns/tags/DwmDns-0.2.7/classes/include/DwmDnsUtils.hh 10129 $
// @(#) $Id: DwmDnsUtils.hh 10129 2018-01-27 04:20:26Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2018
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmDnsUtils.hh
//!  \brief Utility functions for Dwm::Dns
//---------------------------------------------------------------------------

#ifndef _DWMDNSUTILS_HH_
#define _DWMDNSUTILS_HH_

extern "C" {
  #include <sys/types.h>
  #include <sys/socket.h>
  #include <netinet/in.h>
  #include <arpa/inet.h>
}

#include <cstdint>
#include <string>

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
bool operator == (const in6_addr & addr1, const in6_addr & addr2);

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
bool operator < (const in6_addr & addr1, const in6_addr & addr2);

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
bool operator == (const in_addr & addr1, const in_addr & addr2);

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
bool operator < (const in_addr & addr1, const in_addr & addr2);


namespace Dwm {

  namespace Dns {

    //------------------------------------------------------------------------
    //!  Encodes the given string @c into buffer @c pkt of length @c pktlen
    //!  as a DNS 'character string' starting at @c ptr.  Returns the address
    //!  immediately following the encoded character string in @c pkt on
    //!  success.  Throws std::out_of_range if the encoded character string
    //!  will not fit in @c pkt.
    //------------------------------------------------------------------------
    uint8_t *EncodeCharacterString(const std::string & s, uint8_t *pkt,
                                   uint8_t *ptr, uint16_t pktlen);
    
    //------------------------------------------------------------------------
    //!  Decodes a DNS 'character string' from buffer @c pkt of length
    //!  @c pktlen into @c s, starting at @c ptr.  Returns the address
    //!  immediately following the encoded character string in @c pkt on
    //!  success.  Throws std::out_of_range if @c pkt was too short to
    //!  contain the encoded character string.
    //------------------------------------------------------------------------
    const uint8_t *DecodeCharacterString(std::string & s, const uint8_t *pkt,
                                         const uint8_t *ptr, uint16_t pktlen);

    //------------------------------------------------------------------------
    //!  Converts the given IP address @c ipAddr into @c arpa in ARPA domain
    //!  form.  ARPA form is used for PTR lookups.  Returns true on success,
    //!  false on failure.
    //------------------------------------------------------------------------
    bool ToArpa(const std::string & ipAddr, std::string & arpa);

    //------------------------------------------------------------------------
    //!  Converts the given IPv4 address @c inAddr into @c arpa in ARPA
    //!  domain form.  ARPA form is used for PTR lookups.
    //------------------------------------------------------------------------
    void ToArpa(const in_addr & inAddr, std::string & arpa);

    //------------------------------------------------------------------------
    //!  Converts the given IPv6 address @c in6Addr into @c arpa in ARPA
    //!  domain form.  ARPA form is used for PTR lookups.
    //------------------------------------------------------------------------
    void ToArpa(const in6_addr & in6Addr, std::string & arpa);
    
    //------------------------------------------------------------------------
    //!  Converts the given ARPA domain @c arpa into @c ipAddr.  Returns true
    //!  on success, false on failure.
    //------------------------------------------------------------------------
    bool FromArpa(const std::string & arpa, std::string & ipAddr);
    
  }  // namespace Dns

}  // namespace Dwm

#endif  // _DWMDNSUTILS_HH_
