//===========================================================================
// @(#) $DwmPath: dwm/DwmDns/tags/DwmDns-0.2.7/classes/src/DwmDnsLabelSequence.cc 10133 $
// @(#) $Id: DwmDnsLabelSequence.cc 10133 2018-01-27 17:41:32Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2000, 2016, 2018
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmDnsLabelSequence.cc
//!  \brief Dwm::Dns::LabelSequence class implementation
//---------------------------------------------------------------------------

extern "C" {
  #include <arpa/inet.h>
}

#include <cassert>
#include <cstring>
#include <stdexcept>

#include "DwmSvnTag.hh"
#include "DwmStringUtils.hh"
#include "DwmDnsLabel.hh"
#include "DwmDnsLabelSequence.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/DwmDns/tags/DwmDns-0.2.7/classes/src/DwmDnsLabelSequence.cc 10133 $");

using namespace std;

namespace Dwm {

  namespace Dns {

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    LabelSequence::LabelSequence()
        : _value()
    {}
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    LabelSequence::LabelSequence(const std::string & value)
        : _value(value)
    {}
    
    //------------------------------------------------------------------------
    //!
    //------------------------------------------------------------------------
    LabelSequence::operator std::string () const
    {
      return _value;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint8_t *LabelSequence::Encode(uint8_t *pkt, uint8_t *ptr,
                                   uint16_t pktlen) const
    {
      uint8_t  *ourptr = ptr;
      vector<string>  labelStrings;
      StringUtils::StringToVector(_value, '.', labelStrings);
      for (auto & labelString : labelStrings) {
        Label  label(labelString);
        ptr = label.Encode(pkt, ptr, pktlen);
      }
      if (ptr <= (pkt + pktlen)) {
        *ptr++ = 0;
      }
      else {
        throw out_of_range("Dwm::Dns::LabelSequence will not fit in packet");
      }
      return ptr;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint8_t *LabelSequence::Encode(uint8_t *pkt, uint8_t *ptr,
                                   uint16_t pktlen,
                                   LabelPositions & lps) const
    {
      uint8_t  *ourptr = ptr;
      pair<string,uint16_t>  longestPrev = lps.FindLongestPrev(_value);
      if (longestPrev.second != 0xFFFF) {
        string::size_type  eidx = _value.find(longestPrev.first);
        if (eidx > 0) {
          vector<pair<string,uint16_t>> labelVec;
          vector<string>  labelStrings;
          StringUtils::StringToVector(_value.substr(0, eidx), '.',
                                      labelStrings);
          for (auto lsi = labelStrings.begin();
               lsi != labelStrings.end(); ++lsi) {
            Label  label(*lsi);
            labelVec.push_back(pair<string,uint16_t>(*lsi, ptr - pkt));
            ptr = label.Encode(pkt, ptr, pktlen);
          }
          lps.Add(labelVec, longestPrev.first);
        }
        if ((ptr + 2) <= (pkt + pktlen)) {
          *ptr++ = (longestPrev.second >> 8) | 0xc0;
          *ptr++ = longestPrev.second & 0xFF;
        }
        else {
          throw out_of_range("Dwm::Dns::LabelSequence will not fit in packet");
        }
      }
      else {
        lps.Add(_value, ourptr - pkt);
        vector<pair<string,uint16_t>> labelVec;
        vector<string>  labelStrings;
        StringUtils::StringToVector(_value, '.', labelStrings);
        for (auto & labelString : labelStrings) {
          Label  label(labelString);
          labelVec.push_back(pair<string,uint16_t>(labelString, ptr - pkt));
          ptr = label.Encode(pkt, ptr, pktlen);
        }
        lps.Add(labelVec);
        if (ptr <= (pkt + pktlen)) {
          *ptr++ = 0;
        }
        else {
          throw out_of_range("Dwm::Dns::LabelSequence will not fit in packet");
        }
      }
      return ptr;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const uint8_t *
    LabelSequence::Decode(const uint8_t *pkt, const uint8_t *ptr,
                          uint16_t pktlen)
    {
      _value.clear();
      if (pkt && ptr && pktlen) {
        while (((*ptr & 0xc0) == 0)
               && ((*ptr & 0x3f) > 0)) {
          Label  label;
          ptr = label.Decode(pkt, ptr, pktlen);
          _value += (string)label;
          _value += '.';
        }
        if (_value.size() > 0) {
          _value.pop_back();
        }
        if ((*ptr & 0xc0) == 0xc0) {
          if ((ptr + 2) <= (pkt + pktlen)) {
            uint16_t  offset = (*ptr++ & 0x3f) << 8;
            offset |= *ptr++;
            LabelSequence  ls;
            ls.Decode(pkt, pkt + offset, pktlen);
            if (! _value.empty()) {
              _value += '.';
            }
            _value += ls._value;
          }
          else {
            throw out_of_range("packet too short to contain Dwm::Dns::LabelSequence");
          }
        }
        else {
          if (ptr <= (pkt + pktlen)) {
            ++ptr;
          }
          else {
            throw out_of_range("packet too short to contain Dwm::Dns::LabelSequence");
          }
        }
      }
      return ptr;
    }
    
    
  }  // namespace Dns

}  // namespace Dwm
