//===========================================================================
// @(#) $DwmPath: dwm/DwmDns/tags/DwmDns-0.2.7/classes/src/DwmDnsRRDataCAA.cc 10075 $
// @(#) $Id: DwmDnsRRDataCAA.cc 10075 2018-01-25 02:59:37Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2018
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmDnsRRDataCAA.cc
//!  \brief Dwm::Dns::RRDataCAA class implementation
//---------------------------------------------------------------------------

#include <cstring>
#include <stdexcept>

#include "DwmSvnTag.hh"
#include "DwmDnsRRDataCAA.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/DwmDns/tags/DwmDns-0.2.7/classes/src/DwmDnsRRDataCAA.cc 10075 $");

using namespace std;

namespace Dwm {

  namespace Dns {
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    RRDataCAA::RRDataCAA()
        : _flags(0), _tag(), _value()
    {}
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    RRDataCAA::RRDataCAA(uint8_t flags, const string & tag,
                         const string & value)
        : _flags(flags), _tag(tag), _value(value)
    {}
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool RRDataCAA::operator == (const RRDataCAA & caa) const
    {
      return ((_flags == caa._flags)
              && (_tag == caa._tag)
              && (_value == caa._value));
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint8_t RRDataCAA::Flags() const
    {
      return _flags;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint8_t RRDataCAA::Flags(uint8_t flags)
    {
      _flags = flags;
      return _flags;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const string & RRDataCAA::Tag() const
    {
      return _tag;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const string & RRDataCAA::Tag(const string & tag)
    {
      _tag = tag;
      return _tag;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const string & RRDataCAA::Value() const
    {
      return _value;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const string & RRDataCAA::Value(const string & value)
    {
      _value = value;
      return _value;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint8_t *RRDataCAA::Encode(uint8_t *pkt, uint8_t *ptr,
                               uint16_t pktlen, LabelPositions & lps) const
    {
      if ((ptr + 2) <= (pkt + pktlen)) {
        *ptr++ = _flags;
        uint8_t  len = _tag.size() & 0xFF;
        *ptr++ = len;
        if ((ptr + len) <= (pkt + pktlen)) {
          if (len > 0) {
            memcpy(ptr, _tag.c_str(), len);
            ptr += len;
          }
        }
        else {
          throw out_of_range("Dns::RRDataCAA will not fit in packet");
        }
        if ((ptr + _value.size()) <= (pkt + pktlen)) {
          memcpy(ptr, _value.c_str(), _value.size());
          ptr += _value.size();
        }
        else {
          throw	out_of_range("Dns::RRDataCAA will not fit in packet");
        }
      }
      else {
        throw out_of_range("Dns::RRDataCAA will not fit in packet");
      }
      return ptr;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const uint8_t *RRDataCAA::Decode(const uint8_t *pkt, const uint8_t *ptr,
                                     uint16_t pktlen, uint16_t rdlen)
    {
      _flags = 0;
      _tag.clear();
      _value.clear();

      const uint8_t  *origptr = ptr;
      if ((ptr + 2) <= (pkt + pktlen)) {
        _flags = *ptr++;
        uint8_t  taglen = *ptr++;
        if (taglen > 0) {
          if ((ptr + taglen) <= (pkt + pktlen)) {
            _tag.assign((const char *)ptr, taglen);
            ptr += taglen;
          }
          else {
            throw out_of_range("packet too short to contain"
                               " Dwm::Dns::RRDataCAA");
          }
        }
        int  bytesRemaining = rdlen - (ptr - origptr);
        if (bytesRemaining > 0) {
          _value.assign((const char *)ptr, bytesRemaining);
          ptr += bytesRemaining;
        }
        else if (bytesRemaining < 0) {
          throw out_of_range("packet too short to contain"
                             " Dwm::Dns::RRDataCAA");
        }
      }
      else {
        throw out_of_range("packet too short to contain Dwm::Dns::RRDataCAA");
      }
      return ptr;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    ostream & operator << (ostream & os, const RRDataCAA & caa)
    {
      if (os) {
        os << (uint16_t)caa._flags << ' ' << caa._tag
           << " \"" << caa._value << '"';
      }
      return os;
    }
    
  }  // namespace Dns

}  // namespace Dwm
