//===========================================================================
// @(#) $DwmPath: dwm/DwmDns/tags/DwmDns-0.2.7/classes/src/DwmDnsRRDataRRSIG.cc 10137 $
// @(#) $Id: DwmDnsRRDataRRSIG.cc 10137 2018-01-28 07:25:17Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2018
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmDnsRRDataRRSIG.cc
//!  \brief Dwm::Dns::RRDataRRSIG class implementation
//---------------------------------------------------------------------------

extern "C" {
  #include <arpa/inet.h>
  #include <time.h>
}

#include <cstring>
#include <map>
#include <stdexcept>

#include "DwmSvnTag.hh"
#include "DwmBase64.hh"
#include "DwmDnsLabelSequence.hh"
#include "DwmDnsResourceRecord.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/DwmDns/tags/DwmDns-0.2.7/classes/src/DwmDnsRRDataRRSIG.cc 10137 $");

using namespace std;

namespace Dwm {

  namespace Dns {

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    RRDataRRSIG::RRDataRRSIG()
        : _signersName(), _signature()
    {
      memset(&_fixedData, 0, sizeof(_fixedData));
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    RRDataRRSIG::RRDataRRSIG(uint16_t typeCovered, uint8_t algorithm,
                             uint8_t labels, uint32_t originalTTL,
                             uint32_t signatureExpiration,
                             uint32_t signatureInception, uint16_t keyTag,
                             const string & signersName,
                             const string & signature)
        : _signersName(signersName), _signature(signature)
    {
      _fixedData.typeCovered = typeCovered;
      _fixedData.algorithm = algorithm;
      _fixedData.labels = labels;
      _fixedData.originalTTL = originalTTL;
      _fixedData.signatureExpiration = signatureExpiration;
      _fixedData.signatureInception = signatureInception;
      _fixedData.keyTag = keyTag;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool RRDataRRSIG::operator == (const RRDataRRSIG & rrsig) const
    {
      return ((memcmp(&_fixedData, &rrsig._fixedData,
                 sizeof(_fixedData)) == 0)
              && (_signersName == rrsig._signersName)
              && (_signature == rrsig._signature));
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint16_t RRDataRRSIG::TypeCovered() const
    {
      return _fixedData.typeCovered;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint16_t RRDataRRSIG::TypeCovered(uint16_t typeCovered)
    {
      _fixedData.typeCovered = typeCovered;
      return _fixedData.typeCovered;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint8_t RRDataRRSIG::Algorithm() const
    {
      return _fixedData.algorithm;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint8_t RRDataRRSIG::Algorithm(uint8_t algorithm)
    {
      _fixedData.algorithm = algorithm;
      return _fixedData.algorithm;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint8_t RRDataRRSIG::Labels() const
    {
      return _fixedData.labels;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint8_t RRDataRRSIG::Labels(uint8_t labels)
    {
      _fixedData.labels = labels;
      return _fixedData.labels;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint32_t RRDataRRSIG::OriginalTTL() const
    {
      return _fixedData.originalTTL;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint32_t RRDataRRSIG::OriginalTTL(uint32_t originalTTL)
    {
      _fixedData.originalTTL = originalTTL;
      return _fixedData.originalTTL;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint32_t RRDataRRSIG::SignatureExpiration() const
    {
      return _fixedData.signatureExpiration;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint32_t RRDataRRSIG::SignatureExpiration(uint32_t signatureExpiration)
    {
      _fixedData.signatureExpiration = signatureExpiration;
      return _fixedData.signatureExpiration;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint32_t RRDataRRSIG::SignatureInception() const
    {
      return _fixedData.signatureInception;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint32_t RRDataRRSIG::SignatureInception(uint32_t signatureInception)
    {
      _fixedData.signatureInception = signatureInception;
      return _fixedData.signatureInception;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint16_t RRDataRRSIG::KeyTag() const
    {
      return _fixedData.keyTag;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint16_t RRDataRRSIG::KeyTag(uint16_t keyTag)
    {
      _fixedData.keyTag = keyTag;
      return _fixedData.keyTag;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const string & RRDataRRSIG::SignersName() const
    {
      return _signersName;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const string & RRDataRRSIG::SignersName(const string & signersName)
    {
      _signersName = signersName;
      return signersName;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const string & RRDataRRSIG::Signature() const
    {
      return _signature;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const string & RRDataRRSIG::Signature(const string & signature)
    {
      _signature = signature;
      return _signature;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint8_t *RRDataRRSIG::Encode(uint8_t *pkt, uint8_t *ptr,
                                 uint16_t pktlen, LabelPositions & lps) const
    {
      if ((ptr + sizeof(_fixedData)) <= (pkt + pktlen)) {
        fixed_data_t  fixed = ToNetworkByteOrder(&_fixedData);
        memcpy(ptr, &fixed, sizeof(fixed));
        ptr += sizeof(fixed);
        LabelSequence  ls(_signersName);
        ptr = ls.Encode(pkt, ptr, pktlen);
        if (! _signature.empty()) {
          if ((ptr + _signature.size()) <= (pkt + pktlen)) {
            memcpy(ptr, _signature.c_str(), _signature.size());
            ptr += _signature.size();
          }
          else {
            throw out_of_range("Dwm::Dns::RRDataRRSIG will not fit in packet");
          }
        }
      }
      return ptr;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const uint8_t *RRDataRRSIG::Decode(const uint8_t *pkt, const uint8_t *ptr,
                                       uint16_t pktlen, uint16_t rdlen)
    {
      memset(&_fixedData, 0, sizeof(_fixedData));
      _signersName.clear();
      _signature.clear();
      
      if ((ptr + sizeof(_fixedData)) <= (pkt + pktlen)) {
        fixed_data_t  fixed;
        memcpy(&fixed, ptr, sizeof(fixed));
        ptr += sizeof(fixed);
        _fixedData = ToHostByteOrder(&fixed);
        int  bytesRemaining = rdlen - sizeof(_fixedData);
        if (bytesRemaining > 0) {
          const uint8_t  *lsptr = ptr;
          LabelSequence  ls;
          ptr = ls.Decode(pkt, ptr, pktlen);
          _signersName = (string)ls;
          bytesRemaining -= (ptr - lsptr);
          if (bytesRemaining > 0) {
            _signature.assign((const char *)ptr, bytesRemaining);
            ptr += bytesRemaining;
          }
        }
      }
      else {
        throw out_of_range("packet too short to contain Dwm::Dns::RRDataRRSIG");
      }
      return ptr;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    ostream & operator << (ostream & os, const RRDataRRSIG & rrsig)
    {
      if (os) {
        os << ResourceRecord::TypeName(rrsig._fixedData.typeCovered) << ' '
           << RRDataRRSIG::AlgorithmName(rrsig._fixedData.algorithm) << ' '
           << (uint16_t)rrsig._fixedData.labels << ' '
           << rrsig._fixedData.originalTTL << ' ';
        struct tm  tm;
        time_t     t = rrsig._fixedData.signatureExpiration;
        gmtime_r(&t, &tm);
        char       tmbuf[32] = { 0 };
        strftime(tmbuf, sizeof(tmbuf), "%Y%m%d%H%M%S", &tm);
        os << tmbuf << ' ';
        t = rrsig._fixedData.signatureInception;
        gmtime_r(&t, &tm);
        memset(&tmbuf, 0, sizeof(tmbuf));
        strftime(tmbuf, sizeof(tmbuf), "%Y%m%d%H%M%S", &tm);
        os << tmbuf << ' '
           << rrsig._fixedData.keyTag << ' '
           << rrsig._signersName << ' '
           << Base64::Encode(rrsig._signature);
      }
      return os;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    string RRDataRRSIG::AlgorithmName(uint8_t algo)
    {
      string  rc;
      static const map<uint8_t,string>  algoNames = {
        { k_algoRSAMD5,     "RSAMD5"     },
        { k_algoDH,         "DH"         },
        { k_algoDSA,        "DSA"        },
        { k_algoECC,        "ECC"        },
        { k_algoRSASHA1,    "RSASHA1"    },
        { k_algoINDIRECT,   "INDIRECT"   },
        { k_algoPRIVATEDNS, "PRIVATEDNS" },
        { k_algoPRIVATEOID, "PRIVATEOID" }
      };
      auto  it = algoNames.find(algo);
      if (it != algoNames.end()) {
        rc = it->second;
      }
      else {
        rc = to_string((uint16_t)algo);
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    RRDataRRSIG::fixed_data_t
    RRDataRRSIG::ToNetworkByteOrder(const RRDataRRSIG::fixed_data_t *data)
    {
      fixed_data_t  rc = *data;
      rc.typeCovered = htons(rc.typeCovered);
      rc.originalTTL = htonl(rc.originalTTL);
      rc.signatureExpiration = htonl(rc.signatureExpiration);
      rc.signatureInception = htonl(rc.signatureInception);
      rc.keyTag = htons(rc.keyTag);
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    RRDataRRSIG::fixed_data_t
    RRDataRRSIG::ToHostByteOrder(const RRDataRRSIG::fixed_data_t *data)
    {
      fixed_data_t  rc = *data;
      rc.typeCovered = ntohs(rc.typeCovered);
      rc.originalTTL = ntohl(rc.originalTTL);
      rc.signatureExpiration = ntohl(rc.signatureExpiration);
      rc.signatureInception = ntohl(rc.signatureInception);
      rc.keyTag = ntohs(rc.keyTag);
      return rc;
    }
    

  }  // namespace Dns
  
}  // namespace Dwm
