//===========================================================================
// @(#) $DwmPath: dwm/DwmDns/trunk/classes/src/DwmDnsRRDataDNSKEY.cc 10133 $
// @(#) $Id: DwmDnsRRDataDNSKEY.cc 10133 2018-01-27 17:41:32Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2018
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmDnsRRDataDNSKEY.cc
//!  \brief Dwm::Dns::RRDataDNSKEY class implementation
//---------------------------------------------------------------------------

extern "C" {
  #include <arpa/inet.h>
}

#include <cstring>

#include "DwmSvnTag.hh"
#include "DwmBase64.hh"
#include "DwmDnsRRDataDNSKEY.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/DwmDns/trunk/classes/src/DwmDnsRRDataDNSKEY.cc 10133 $");

using namespace std;

namespace Dwm {

  namespace Dns {

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    RRDataDNSKEY::RRDataDNSKEY()
        : _flags(0), _protocol(0), _algorithm(0), _publicKey()
    {}
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    RRDataDNSKEY::RRDataDNSKEY(uint16_t flags, uint8_t protocol,
                               uint8_t algorithm, const string & publicKey)
        : _flags(flags), _protocol(protocol), _algorithm(algorithm),
          _publicKey(publicKey)
    {}
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool RRDataDNSKEY::operator == (const RRDataDNSKEY & dnskey) const
    {
      bool  rc = false;
      if ((_flags == dnskey._flags)
          && (_protocol == dnskey._protocol)
          && (_algorithm == dnskey._algorithm)
          && (_publicKey == dnskey._publicKey)) {
        rc = true;
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint16_t RRDataDNSKEY::Flags() const
    {
      return _flags;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint16_t RRDataDNSKEY::Flags(uint16_t flags)
    {
      _flags = flags;
      return _flags;
    }
      
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint8_t RRDataDNSKEY::Protocol() const
    {
      return _protocol;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint8_t RRDataDNSKEY::Protocol(uint8_t protocol)
    {
      _protocol = protocol;
      return _protocol;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint8_t RRDataDNSKEY::Algorithm() const
    {
      return _algorithm;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint8_t RRDataDNSKEY::Algorithm(uint8_t algorithm)
    {
      _algorithm = algorithm;
      return _algorithm;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const string & RRDataDNSKEY::PublicKey() const
    {
      return _publicKey;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const string & RRDataDNSKEY::PublicKey(const string & publicKey)
    {
      _publicKey = publicKey;
      return _publicKey;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint8_t *RRDataDNSKEY::Encode(uint8_t *pkt, uint8_t *ptr,
                                  uint16_t pktlen, LabelPositions & lps) const
    {
      if ((ptr + 4) <= (pkt + pktlen)) {
        uint16_t  flags = htons(_flags);
        memcpy(ptr, &flags, sizeof(flags));
        ptr += sizeof(flags);
        *ptr++ = _protocol;
        *ptr++ = _algorithm;
        if (_publicKey.size() > 0) {
          if ((ptr + _publicKey.size()) <= (pkt + pktlen)) {
            memcpy(ptr, _publicKey.c_str(), _publicKey.size());
            ptr += _publicKey.size();
          }
          else {
            throw out_of_range("Dwn::Dns::RRDataDNSKEY will not fit in packet");
          }
        }
      }
      else {
        throw out_of_range("Dwn::Dns::RRDataDNSKEY will not fit in packet");
      }
      return ptr;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const uint8_t *RRDataDNSKEY::Decode(const uint8_t *pkt, const uint8_t *ptr,
                                        uint16_t pktlen, uint16_t rdlen)
    {
      _flags = 0;
      _protocol = 0;
      _algorithm = 0;
      _publicKey.clear();
      
      uint16_t        bytesLeft = rdlen;
      const uint8_t  *origptr = ptr;
      
      if ((ptr + 4) <= (pkt + pktlen)) {
        uint16_t  flags;
        memcpy(&flags, ptr, sizeof(flags));
        ptr += sizeof(flags);
        _flags = ntohs(flags);
        _protocol = *ptr++;
        _algorithm = *ptr++;
        bytesLeft -= (ptr - origptr);
        if (bytesLeft) {
          _publicKey.assign((const char *)ptr, bytesLeft);
          ptr += bytesLeft;
        }
      }
      return ptr;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    ostream & operator << (ostream & os, const RRDataDNSKEY & dnskey)
    {
      if (os) {
        os << dnskey._flags << ' ' << (uint16_t)dnskey._protocol << ' '
           << (uint16_t)dnskey._algorithm << ' '
           << Base64::Encode(dnskey._publicKey);
      }
      return os;
    }
    
  }  // namespace Dns

}  // namespace Dwm
