//===========================================================================
// @(#) $Name:$
// @(#) $Id: DwmCfg2Json.hh 9088 2017-04-14 10:24:13Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2017
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmCfg2Json.hh
//!  \brief Simple configuration to JSON converter class definition
//---------------------------------------------------------------------------

#include <iostream>
#include <string>
#include <vector>
#include <json/json.h>  // jsoncpp header file

#ifndef _DWMCFG2JSON_HH_
#define _DWMCFG2JSON_HH_

namespace Dwm {

  //--------------------------------------------------------------------------
  //!  Simple class to convert a configuration file in my preferred style
  //!  to a Json::Value.  The main intent of this class: allow me to avoid
  //!  writing new lexers/parsers for configuration files.
  //--------------------------------------------------------------------------
  class Cfg2Json
  {
  public:
    //------------------------------------------------------------------------
    //!  Constructor.
    //------------------------------------------------------------------------
    Cfg2Json();
    
    //------------------------------------------------------------------------
    //!  Parses the config file named @c confFilename.  Returns true on
    //!  success, false on failure.
    //------------------------------------------------------------------------
    bool Parse(const std::string & confFilename);
    
    //------------------------------------------------------------------------
    //!  Returns the encapsulated Json::Value.
    //------------------------------------------------------------------------
    const Json::Value & JsonValue() const;

    //------------------------------------------------------------------------
    //!  Sets and returns the encapsulated Json::Value.
    //------------------------------------------------------------------------
    const Json::Value & JsonValue(const Json::Value & value);

    //------------------------------------------------------------------------
    //!  If a Json::Value is found at the given @c path, returns a const
    //!  pointer to it.  Else returns null.  @c path is a slash-separated
    //!  path much like a filesystem path.  See tests/TestCfg2Json.cc and
    //!  tests/test.conf from the source distribution for example usage.
    //------------------------------------------------------------------------
    const Json::Value * Find(const std::string & path) const;

    //------------------------------------------------------------------------
    //!  Finds the Json::Value at the given @c path, relative to the given
    //!  value @c jvp.  On success, returns a pointer to the found node.
    //!  Returns null if no node is found.
    //------------------------------------------------------------------------
    const Json::Value * Find(const Json::Value *jvp,
                             const std::string & path) const;

    //------------------------------------------------------------------------
    //!  Convert the encapsulated Json::Value back into a string in
    //!  configuration file form.
    //------------------------------------------------------------------------
    std::string ToConfString() const;
    
  private:
    Json::Value  _jsonValue;

    static void ToConfString(std::ostream & os, const Json::Value & jv,
                             int indent = 0);
    static std::vector<std::string> GetPathComps(const std::string & path);
  };
  
}  // namespace Dwm

#endif  // _DWMCFG2JSON_HH_
