//===========================================================================
// @(#) $Name:$
// @(#) $Id: TestCfg2Json.cc 9087 2017-04-14 09:36:02Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2017
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file TestCfg2Json.cc
//!  \brief Unit tests for Dwm::Cfg2Json class
//---------------------------------------------------------------------------

extern "C" {
  #include <unistd.h>
}

#include <iostream>
#include <fstream>

#include "DwmUnitAssert.hh"
#include "DwmCfg2Json.hh"

using namespace std;
using namespace Dwm;

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static void TestSyslogStanza(const Json::Value & jv)
{
  if (UnitAssert(jv.isMember("Syslog"))) {
    UnitAssert(jv["Syslog"].size() == 3);
    if (UnitAssert(jv["Syslog"].isMember("Facility"))) {
      UnitAssert(jv["Syslog"]["Facility"].size() == 0);
      if (UnitAssert(jv["Syslog"]["Facility"].isString())) {
        UnitAssert(jv["Syslog"]["Facility"].asString() == "local0");
      }
    }
    if (UnitAssert(jv["Syslog"].isMember("MinimumPriority"))) {
      UnitAssert(jv["Syslog"]["MinimumPriority"].size() == 0);
      if (UnitAssert(jv["Syslog"]["MinimumPriority"].isString())) {
        UnitAssert(jv["Syslog"]["MinimumPriority"].asString() == "debug");
      }
    }
    if (UnitAssert(jv["Syslog"].isMember("ShowFileLocations"))) {
      UnitAssert(jv["Syslog"]["ShowFileLocations"].size() == 0);
      if (UnitAssert(jv["Syslog"]["ShowFileLocations"].isString())) {
        UnitAssert(jv["Syslog"]["ShowFileLocations"].asString() == "true");
      }
    }
  }
  return;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static void TestDeeplyNestedEmptyStanza(const Json::Value & jv)
{
  const Json::Value  *jvp = 0;
  
  static const vector<string>  names = {
    "Deeply", "Nested", "Stanzas", "Should",
    "Work", "Even", "If", "Empty", ""
  };
  const Json::Value *jvr = &jv;
  for (auto n : names) {
    if (! n.empty()) {
      if (jvr != &jv) {
        UnitAssert(jvr->size() == 1);
      }
      if (UnitAssert(jvr->isMember(n))) {
        jvr = &((*jvr)[n]);
      }
      else {
        break;
      }
    }
    else {
      UnitAssert(jvr->isNull());
    }
  }
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static bool TestServerStanza(const Json::Value & jv, const string & name,
                             const string & addr, const string & port)
{
  bool  rc = false;
  if (! UnitAssert(jv.isMember(name))) {
    goto done;
  }
  UnitAssert(jv[name].size() == 2);
  if (! UnitAssert(jv[name].isMember("Address"))) {
    goto done;
  }
  UnitAssert(jv[name]["Address"].asString() == addr);
  if (! UnitAssert(jv[name].isMember("Port"))) {
    goto done;
  }
  rc = UnitAssert(jv[name]["Port"].asString() == port);
done:
  return rc;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static void TestNestedQuoted(const Json::Value & jv)
{
  if (! UnitAssert(jv.isMember("Nested With Quotes"))) {
    goto done;
  }
  UnitAssert(jv["Nested With Quotes"].size() == 3);

  if (! UnitAssert(jv["Nested With Quotes"].isMember("foo"))) {
    goto done;
  }
  if (UnitAssert(jv["Nested With Quotes"]["foo"].isString())) {
    UnitAssert(jv["Nested With Quotes"]["foo"].asString() == "foo");
  }

  if (! UnitAssert(jv["Nested With Quotes"].isMember("foo"))) {
    goto done;
  }
  if (UnitAssert(jv["Nested With Quotes"]["bar"].isString())) {
    UnitAssert(jv["Nested With Quotes"]["bar"].asString() == "bar");
  }

  if (! UnitAssert(jv["Nested With Quotes"].isMember("foo bar"))) {
    goto done;
  }
  UnitAssert(jv["Nested With Quotes"]["foo bar"].size() == 2);
  if (UnitAssert(jv["Nested With Quotes"]["foo bar"].isMember("foo"))) {
    if (UnitAssert(jv["Nested With Quotes"]["foo bar"]["foo"].isString())) {
      UnitAssert(jv["Nested With Quotes"]["foo bar"]["foo"].asString()
                 == "foo");
    }
  }
  if (UnitAssert(jv["Nested With Quotes"]["foo bar"].isMember("bar"))) {
    if (UnitAssert(jv["Nested With Quotes"]["foo bar"]["bar"].isString())) {
      UnitAssert(jv["Nested With Quotes"]["foo bar"]["bar"].asString()
                 == "bar");
    }
  }
  
done:
  return;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static void TestConcatStrings(const Cfg2Json & cfg)
{
  const Json::Value  *jvp;
  if (UnitAssert((jvp = cfg.Find("ConcatStrings/concat")))) {
    if (UnitAssert(jvp->isString())) {
      UnitAssert(jvp->asString() ==
                 "This is a long value, to test concatenation of string"
                 " values. This likely won't see frequent use, but it was"
                 " trivial to add to the parser.");
    }
  }
  return;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static void TestServers(const Json::Value & jv)
{
  if (UnitAssert(jv.isMember("Servers"))) {
    UnitAssert(jv["Servers"].size() == 3);
    UnitAssert(TestServerStanza(jv["Servers"], "telnet", "10.1.2.3", "23"));
    UnitAssert(TestServerStanza(jv["Servers"], "mirai", "10.2.3.4", "5358"));
    UnitAssert(TestServerStanza(jv["Servers"], "mirai2", "10.3.4.5", "7547"));
  }
  return;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static void TestFind(const Cfg2Json & cfg2json)
{
  const Json::Value  *jvp =
    cfg2json.Find("Deeply/Nested/Stanzas/Should/Work/Even/If/Empty");
  if (UnitAssert(jvp)) {
    UnitAssert(jvp->isNull());
  }
  if (UnitAssert((jvp = cfg2json.Find("Servers")))) {
    UnitAssert(jvp->size() == 3);
  }
  if (UnitAssert((jvp = cfg2json.Find("Servers/telnet/Address")))) {
    if (UnitAssert(jvp->isString())) {
      UnitAssert(jvp->asString() == "10.1.2.3");
    }
  }
  if (UnitAssert((jvp = cfg2json.Find("Servers")))) {
    if (UnitAssert((jvp = cfg2json.Find(jvp, "telnet")))) {
      if (UnitAssert((jvp = cfg2json.Find(jvp, "Address")))) {
        if (UnitAssert(jvp->isString())) {
          UnitAssert(jvp->asString() == "10.1.2.3");
        }
      }
    }
  }
                   
  UnitAssert(! cfg2json.Find("Servers/nonexistent"));

  return;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static void TestToConfString(const Cfg2Json & cfg2json)
{
  ofstream  os("/tmp/crap.conf");
  if (UnitAssert(os)) {
    os << cfg2json.ToConfString();
    os.close();
    Cfg2Json  cfg2json2;
    UnitAssert(cfg2json2.Parse("/tmp/crap.conf"));
    UnitAssert(cfg2json2.JsonValue() == cfg2json.JsonValue());
    remove("/tmp/crap.conf");
  }
  
  return;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
int main(int argc, char *argv[])
{
  string  confFile("./test.conf");

  int  optChar;
  while ((optChar = getopt(argc, argv, "f:")) != EOF) {
    switch (optChar) {
      case 'f':
        confFile = optarg;
        break;
      default:
        break;
    }
  }
  
  Cfg2Json  cfg2json;
  const Json::Value & jv = cfg2json.JsonValue();

  if (UnitAssert(cfg2json.Parse(confFile))) {
    UnitAssert(jv.size() == 5);
    TestSyslogStanza(jv);
    TestServers(jv);
    TestNestedQuoted(jv);
    TestDeeplyNestedEmptyStanza(jv);
    TestConcatStrings(cfg2json);
    TestFind(cfg2json);
    TestToConfString(cfg2json);
  }
  
  if (Assertions::Total().Failed() > 0) {
    Assertions::Print(cerr, true);
    return(1);
  }
  else {
    cout << Assertions::Total() << " passed\n";
  }

}
