//===========================================================================
// @(#) $Name:$
// @(#) $Id: DwmPfDevice.hh 9387 2017-05-24 02:59:38Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2017
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmPfDevice.hh
//!  \brief Dwm::Pf::Device class definition
//---------------------------------------------------------------------------

#ifndef _DWMPFDEVICE_HH_
#define _DWMPFDEVICE_HH_

#include <map>
#include <mutex>
#include <string>
#include <vector>

#include "DwmPfRule.hh"
#include "DwmPfTable.hh"

namespace Dwm {

  namespace Pf {

    //------------------------------------------------------------------------
    //!  Encapsulates the pf device.
    //------------------------------------------------------------------------
    class Device
    {
    public:
      //----------------------------------------------------------------------
      //!  Construct from the path to the pf device (usually "/dev/pf").
      //----------------------------------------------------------------------
      Device(const std::string & path);
      
      //----------------------------------------------------------------------
      //!  Destructor.  If the device is open, it will be closed.
      //----------------------------------------------------------------------
      ~Device();
      
      //----------------------------------------------------------------------
      //!  Opens the pf device.  Returns true on success, false on failure.
      //----------------------------------------------------------------------
      bool Open();
      
      //----------------------------------------------------------------------
      //!  Closes the pf device.
      //----------------------------------------------------------------------
      void Close();

      //----------------------------------------------------------------------
      //!  Returns true is the device is open.
      //----------------------------------------------------------------------
      bool IsOpen() const;
      
      //----------------------------------------------------------------------
      //!  Fetch all tables.
      //----------------------------------------------------------------------
      std::vector<Table> GetTables() const;
      
      //----------------------------------------------------------------------
      //!  Fetch a table by anchor and table name.
      //----------------------------------------------------------------------
      Table GetTable(const std::string & anchor,
                     const std::string & name) const;
      
      //----------------------------------------------------------------------
      //!  Returs true if the given @c table exists.
      //----------------------------------------------------------------------
      bool TableExists(const Table & table) const;
      
      //----------------------------------------------------------------------
      //!  Creates a table of the given @c name under the given @c anchor.
      //!  Returns true on success, false on failure.
      //----------------------------------------------------------------------
      bool CreateTable(const std::string & anchor,
                       const std::string & name) const;
      
      //----------------------------------------------------------------------
      //!  Deletes the table of the given @c name under the given @c anchor.
      //!  Returns true on success, false on failure.
      //----------------------------------------------------------------------
      bool DeleteTable(const std::string & anchor,
                       const std::string & name) const;

      //----------------------------------------------------------------------
      //!  Returns all rules.  EXPERIMENTAL
      //----------------------------------------------------------------------
      std::vector<Rule> GetRules() const;

      //----------------------------------------------------------------------
      //!  Returns rules which use the table named @c tableName.
      //!  EXPERIMENTAL
      //----------------------------------------------------------------------
      std::vector<Rule> GetRulesWithTable(const std::string & tableName) const;
      

      //----------------------------------------------------------------------
      //!  Calls ioctl() on the pf device with the given request @c req and
      //!  data @c v.  Returns true on success, false on failure.
      //----------------------------------------------------------------------
      bool Ioctl(unsigned long req, void *v) const;
      
      //----------------------------------------------------------------------
      //!  Conversion operator that just returns our encapsulated file
      //!  descriptor.  This is just convenient for checking that the
      //!  device is open (descriptor greater than or equal to 0).
      //----------------------------------------------------------------------
      // operator int () const;
      
    private:
      std::string         _path;
      int                 _fd;
      mutable std::mutex  _mtx;

      static const std::map<unsigned long, std::string>  _ioctlNames;
      static std::string IoctlReqName(unsigned long r);
    };
    
  }  // namespace Pf

}  // namespace Dwm

#endif  // _DWMPFDEVICE_HH_
