//===========================================================================
// @(#) $Name:$
// @(#) $Id: dwmrdapc.cc 11028 2020-08-28 06:56:02Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2017
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file dwmrdapc.cc
//!  \brief dwmrdap client, mostly just an example
//---------------------------------------------------------------------------

extern "C" {
  #include <sys/types.h>
  #include <sys/socket.h>
  #include <sys/stat.h>
  #include <netinet/in.h>
  #include <netinet/tcp.h>
  #include <arpa/inet.h>
  #include <fcntl.h>
  #include <netdb.h>
  #include <pwd.h>
  #include <unistd.h>
}

#include <cstdlib>
#include <fstream>
#include <regex>
#include <sstream>
#include <string>

#include "DwmIO.hh"
#include "DwmIpv4Prefix.hh"
#include "DwmOptArgs.hh"
#include "DwmSocket.hh"
#include "DwmSvnTag.hh"
#include "DwmSysLogger.hh"
#include "DwmAuthPeerAuthenticator.hh"
#include "DwmRDAPRequestMessage.hh"
#include "DwmRDAPResponseMessage.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/libDwmRDAP/tags/libDwmRDAP-0.2.1/apps/dwmrdapc/dwmrdapc.cc 11028 $");

using namespace std;
using namespace Dwm;

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static string GetHomeDirectory()
{
  string  rc;
  uid_t   uid = geteuid();
  struct passwd *pw = getpwuid(uid);
  if (pw) {
    rc = pw->pw_dir;
  }
  return rc;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static string GetKeyDirectory()
{
  string  rc;
  string  homeDir = GetHomeDirectory();
  if (! homeDir.empty()) {
    rc = homeDir + "/.dwmauth";
  }
  return rc;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static bool Authenticate(Dwm::Socket & s, string & theirId, string & agreedKey)
{
  bool  rc = false;
  Auth::PeerAuthenticator  peerAuth(GetKeyDirectory() + "/id_ed25519",
                                    GetKeyDirectory() + "/known_services");
  if (peerAuth.Authenticate(s, theirId, agreedKey)) {
    rc = true;
  }
  return rc;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
string RequestResponse(Socket & s, const string & secret,
                       const RDAP::RequestMessage & request)
{
  string                  rc;
  RDAP::ResponseMessage   response;
  Json::FastWriter        jfw;
  if (request.Write(s, secret)) {
    if (response.Read(s, secret)) {
      // rc = jfw.write(response.Json());
      rc = response.Json().toStyledString();
    }
    else {
      Syslog(LOG_ERR, "Failed to read server response");
    }
  }
  else {
    Syslog(LOG_ERR, "Failed to write server request");
  }
  return rc;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
bool RequestResponse(Socket & s, const string & secret,
                     const RDAP::RequestMessage & request,
                     RDAP::ResponseMessage & response)
{
  bool  rc = false;
  if (request.Write(s, secret)) {
    if (response.Read(s, secret)) {
      rc = true;
    }
    else {
      Syslog(LOG_ERR, "Failed to read server response");
    }
  }
  else {
    Syslog(LOG_ERR, "Failed to write server request");
  }
  return rc;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
bool Connect(const string & host, Socket & s)
{
  bool  rc = false;
  struct hostent  *hostEntry = gethostbyname2(host.c_str(), AF_INET);
  if (! hostEntry) {
    cerr << "Host '" << host << "' not found.\n";
    exit(1);
  }
  
  if (s.Open(PF_INET, SOCK_STREAM, 0)) {
    struct sockaddr_in  servAddr;
    memset(&servAddr, 0, sizeof(servAddr));
#ifndef __linux__
    servAddr.sin_len = sizeof(servAddr);
#endif
    servAddr.sin_family = AF_INET;
    servAddr.sin_port = htons(6363);
    servAddr.sin_addr.s_addr = *(in_addr_t *)hostEntry->h_addr_list[0];
    if (s.Connect(servAddr)) {
      int  noDelay = 1;
      s.Setsockopt(IPPROTO_TCP, TCP_NODELAY, &noDelay, sizeof(noDelay));
      rc = true;
    }
  }
  return rc;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static void Usage(const string & argv0)
{
  cerr << "Usage: " << argv0 << " [-h host] IPv4_address\n";
  //     << "       " << argv0 << " -h host createkeys\n";
  return;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static bool Query(Socket & s, const string & secret,
                  const vector<Ipv4Address> & addrs)
{
  bool  rc = false;
  RDAP::RequestMessage  req(addrs);
  string  resp = RequestResponse(s, secret, req);
  if (! resp.empty()) {
    cout << resp << '\n';
    rc = true;
  }
  return rc;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static bool Delete(Socket & s, const string & secret,
                   const vector<Ipv4Prefix> & prefixes)
{
  bool  rc = false;
  return rc;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
Ipv4Address NextWalkAddress(const Ipv4Prefix & prev, Ipv4Prefix & resp)
{
  Ipv4Address  addr;
  if (resp.MaskLength() < 9) {
    addr = prev.FirstAddress();
    --addr;
    resp = Ipv4Prefix(addr, 24);
    // cerr << "addr from short prefix: " << addr << '\n';
  }
  else {
    addr = resp.FirstAddress();
    --addr;
    // cerr << "addr from good prefix: " << addr << '\n';
  }
  return addr;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static void Walk(Socket & s, const string & secret, const Ipv4Prefix & pfx)
{
  vector<Ipv4Address>  addrs;
  addrs.push_back(pfx.LastAddress());
  Ipv4Prefix  nextPfx(pfx);
  while (addrs[0] >= pfx.FirstAddress()) {
    RDAP::RequestMessage   req(addrs);
    RDAP::ResponseMessage  resp;
    bool  rc = RequestResponse(s, secret, req, resp);
    if (rc) {
      if (resp.Json()[0].isMember("prefix")
          && resp.Json()[0]["prefix"].isString()) {
        Ipv4Prefix  respPfx = Ipv4Prefix(resp.Json()[0]["prefix"].asString());
        addrs[0] = NextWalkAddress(nextPfx, respPfx);
        cout << resp.Json().toStyledString() << '\n';
        nextPfx = respPfx;
      }
      else {
        addrs[0] -= 256;
        nextPfx = Ipv4Prefix(addrs[0], 24);
        // addrs[0] = nextPfx.Network();
        cout << "pfx, addr from no response: " << nextPfx.ToShortString()
             << ' ' << addrs[0] << '\n';
      }
    }
    else {
      break;
    }
  }
  return;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
int main(int argc, char *argv[])
{
  int      rc = 1;
  OptArgs  optargs;
  optargs.AddOptArg("d", "delete", false, "", "delete prefixes");
  optargs.AddOptArg("h:", "host", false, "", "server host");
  optargs.AddOptArg("w", "prefix", false, "", "walk a prefix");
  
  int  nextArg = optargs.Parse(argc, argv);
#if 0
  if (nextArg >= argc - 1) {
    Usage(argv[0]);
    exit(1);
  }
#endif
  
  SysLogger::Open("dwmrdapc", LOG_PERROR, LOG_USER);
  SysLogger::MinimumPriority("info");

  string  rdapHost = optargs.Get<string>('h');
  if (rdapHost.empty()) {
    if (getenv("RDAPDHOST")) {
      rdapHost = getenv("RDAPDHOST");
    }
  }
  if (rdapHost.empty()) {
    cerr << "A host is required.  Either set RDAPDHOST in your environment\n"
         << "or use the '-h host' command line option.\n\n";
    Usage(argv[0]);
    exit(1);
  }
  
  string  hostName;
  struct hostent  *hostEntry =
    gethostbyname2(rdapHost.c_str(), AF_INET);
  if (hostEntry && hostEntry->h_name) {
    hostName = hostEntry->h_name;
  }
  else {
    cerr << "Failed to find host " << rdapHost << '\n';
    Usage(argv[0]);
    exit(1);
  }

  Socket  s;
  string  theirId, secret;
  if (Connect(hostName, s)) {
    if (! Authenticate(s, theirId, secret)) {
      cerr << "Failed to authenticate!\n";
      return 1;
    }
  }
  else {
    cerr << "Failed to connect to " << hostName << '\n';
    return 1;
  }
  
  if (argc > nextArg) {
    if (optargs.Get<bool>('d')) {
      vector<Ipv4Prefix>  prefixes;
      for ( ; nextArg < argc; ++nextArg) {
        prefixes.push_back(Ipv4Prefix(argv[nextArg]));
      }
      if (Delete(s, secret, prefixes)) {
        rc = 0;
      }
    }
    else if (optargs.Get<bool>('w')) {
      for ( ; nextArg < argc; ++nextArg) {
        Walk(s, secret, Ipv4Prefix(argv[nextArg]));
      }
    }
    else {
      vector<Ipv4Address>  addrs;
      for ( ; nextArg < argc; ++nextArg) {
        addrs.push_back(Ipv4Address(argv[nextArg]));
      }
      if (Query(s, secret, addrs)) {
        rc = 0;
      }
    }
  }
  else {
    Usage(argv[0]);
    exit(1);
  }

  return rc;
}
