//===========================================================================
// @(#) $Name:$
// @(#) $Id: DwmRDAPServerConfig.cc 9002 2017-04-10 06:39:25Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2017
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmRDAPServerConfig.cc
//!  \brief Dwm::RDAP::ServerConfig class implementation
//---------------------------------------------------------------------------

#include "DwmSvnTag.hh"
#include "DwmRDAPServerConfig.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/libDwmRDAP/tags/libDwmRDAP-0.2.1/apps/dwmrdapd/DwmRDAPServerConfig.cc 9002 $");

using namespace std;

namespace Dwm {

  namespace RDAP {

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    ServerConfig::ServerConfig(const string & fn)
        : _filename(fn)
    {
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool ServerConfig::Parse()
    {
      return _cfg2json.Parse(_filename);
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    Ipv4Address ServerConfig::TCPAddress() const
    {
      Ipv4Address  rc(INADDR_ANY);
      const Json::Value *val = _cfg2json.Find("Server/TCP/Address");
      if (val) {
        if (val->isString() && (val->asString() != "any")) {
          rc = Ipv4Address(val->asString());
        }
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint16_t ServerConfig::TCPPort() const
    {
      uint16_t  rc = 6363;
      const Json::Value *val = _cfg2json.Find("Server/TCP/Port");
      if (val) {
        if (val->isUInt() && (val->asUInt() <= 65535) && (val->asUInt() > 0)) {
          rc = val->asUInt();
        }
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    string ServerConfig::RDAPIpv4BootstrapFile() const
    {
      return ConfString("Server/Bootstrap/rdap_ipv4",
                        "/usr/local/etc/rdap_ipv4.json");
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    string ServerConfig::CountryCodesFile() const
    {
      return ConfString("Server/Bootstrap/country_codes",
                        "/usr/local/etc/country_codes.json");
    }
      
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    string ServerConfig::DatabaseFile() const
    {
      return ConfString("Server/Database", "/usr/local/var/ipv4country.db");
    }
      
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    string ServerConfig::SyslogFacility() const
    {
      return ConfString("Syslog/Facility", "daemon");
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    string ServerConfig::SyslogMinimumPriority() const
    {
      return ConfString("Syslog/MinimumPriority", "info");
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool ServerConfig::SyslogShowFileLocations() const
    {
      return ConfBool("Syslog/ShowFileLocations", true);
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    string ServerConfig::ConfString(const string & path,
                                    const string & deflt) const
    {
      string  rc(deflt);
      const Json::Value *val = _cfg2json.Find(path);
      if (val) {
        if (val->isString() && (! val->asString().empty())) {
          rc = val->asString();
        }
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool ServerConfig::ConfBool(const string & path, bool dflt) const
    {
      bool  rc(dflt);
      const Json::Value *val = _cfg2json.Find(path);
      if (val) {
        if (val->isString() && (! val->asString().empty())) {
          rc = (val->asString() == "true") ? true : false;
        }
      }
      return rc;
    }
      
  }  // namespace RDAP

}  // namespace Dwm
