//===========================================================================
// @(#) $Name:$
// @(#) $Id: DwmCountryCodes.hh 9237 2017-04-25 06:38:54Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2017
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmCountryCodes.hh
//!  \brief Dwm::CountryCodes class definition
//---------------------------------------------------------------------------

#ifndef _DWMCOUNTRYCODES_HH_
#define _DWMCOUNTRYCODES_HH_

#include <fstream>
#include <map>
#include <vector>

#include "DwmReadable.hh"
#include "DwmWritable.hh"

#include "DwmCountryCode.hh"

namespace Dwm {

  //--------------------------------------------------------------------------
  //!  Encapsulates a set of country codes.
  //--------------------------------------------------------------------------
  class CountryCodes
    : public Dwm::Readable, public Dwm::Writable
  {
  public:
    //------------------------------------------------------------------------
    //!  Default constructor
    //------------------------------------------------------------------------
    CountryCodes() = default;
    
    //------------------------------------------------------------------------
    //!  Finds a country code by 2 or 3 letter code.  Returns a filled
    //!  CountryCode on success, an empty CountryCode on failure.
    //------------------------------------------------------------------------
    CountryCode FindByCode(const std::string & code) const;

    std::vector<CountryCode> FindByNameRegExp(const std::string & rgxp) const;
    
    //------------------------------------------------------------------------
    //!  Reads the country codes from a JSON file.  Returns true on success,
    //!  false on failure.
    //------------------------------------------------------------------------
    bool ReadJson(const std::string & fileName);

    //------------------------------------------------------------------------
    //!  Writes the country code to a file in JSON format.  Returns true
    //!  on success, false on failure.
    //------------------------------------------------------------------------
    bool WriteJson(const std::string & fileName) const;

    //------------------------------------------------------------------------
    //!  Reads the country codes from an istream.  Returns the istream.
    //------------------------------------------------------------------------
    std::istream & Read(std::istream & is);
    
    //------------------------------------------------------------------------
    //!  Reads the country codes from a descriptor.  Returns the number of
    //!  bytes read on success, -1 on failure.
    //------------------------------------------------------------------------
    ssize_t Read(int fd);
    
    //------------------------------------------------------------------------
    //!  Reads the country codes from a FILE.  Returns 1 on success,
    //!  0 on failure (fread() semantics).
    //------------------------------------------------------------------------
    size_t Read(FILE *f);
    
    //------------------------------------------------------------------------
    //!  Writes the country codes to an ostream.  Returns the ostream.
    //------------------------------------------------------------------------
    std::ostream & Write(std::ostream & os) const;
    
    //------------------------------------------------------------------------
    //!  Writes the country codes to a file descriptor.  Returns the number of
    //!  bytes written on success, -1 on failure.
    //------------------------------------------------------------------------
    ssize_t Write(int fd) const;
    
    //------------------------------------------------------------------------
    //!  Writes the country codes to a FILE.  Returns 1 on success, 0 on
    //!  failure (fwrite() semantics).
    //------------------------------------------------------------------------
    size_t Write(FILE *f) const;
    
    //------------------------------------------------------------------------
    //!  Returns the number of bytes that should be written if one of the
    //!  Write() members is called.
    //------------------------------------------------------------------------
    uint32_t StreamedLength() const;

    //------------------------------------------------------------------------
    //!  operator ==
    //------------------------------------------------------------------------
    bool operator == (const CountryCodes & countryCodes) const
    {
      return (_a2toCountryCode == countryCodes._a2toCountryCode);
    }

    //------------------------------------------------------------------------
    //!  Returns true if the country codes have not been loaded.
    //------------------------------------------------------------------------
    bool Empty() const
    {
      return _a2toCountryCode.empty();
    }
    
  private:
    std::map<std::string,CountryCode>  _a2toCountryCode;
    std::map<std::string,CountryCode>  _a3toCountryCode;
  };
  
}  // namespace Dwm

#endif  // _DWMCOUNTRYCODES_HH_
