//===========================================================================
// @(#) $Name:$
// @(#) $Id: DwmCountryCodes.cc 8916 2017-03-27 23:56:59Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2017
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmCountryCodes.cc
//!  \brief Dwm::CountryCodes class implementation
//---------------------------------------------------------------------------

#include <regex>

#include <json/json.h>

#include "DwmIO.hh"
#include "DwmSvnTag.hh"
#include "DwmCountryCodes.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/libDwmRDAP/tags/libDwmRDAP-0.2.1/src/DwmCountryCodes.cc 8916 $");

using namespace std;

namespace Dwm {

  //--------------------------------------------------------------------------
  //!  
  //--------------------------------------------------------------------------
  CountryCode CountryCodes::FindByCode(const string & code) const
  {
    CountryCode  cc;
    string       codeUp(code);
    for (auto & c : codeUp) { c = toupper(c); }
    if (codeUp.length() == 2) {
      auto  a2i = _a2toCountryCode.find(codeUp);
      if (a2i != _a2toCountryCode.end()) {
        cc = a2i->second;
      }
    }
    else if (codeUp.length() == 3) {
      auto  a3i = _a3toCountryCode.find(codeUp);
      if (a3i != _a3toCountryCode.end()) {
        cc = a3i->second;
      }
    }
    return cc;
  }

  //--------------------------------------------------------------------------
  //!  
  //--------------------------------------------------------------------------
  vector<CountryCode> CountryCodes::FindByNameRegExp(const string & rgxp) const
  {
    vector<CountryCode>  rc;
    regex   rgx(rgxp, regex::ECMAScript|regex::optimize);
    smatch  sm;
    for (auto & cc : _a2toCountryCode) {
      if (regex_search(cc.second.Name(), sm, rgx)) {
        rc.push_back(cc.second);
      }
    }
    return rc;
  }

  //--------------------------------------------------------------------------
  //!  
  //--------------------------------------------------------------------------
  bool CountryCodes::ReadJson(const string & fileName)
  {
    _a2toCountryCode.clear();
    _a3toCountryCode.clear();
    
    ifstream  is(fileName.c_str());
    if (is) {
      Json::Reader  jr;
      Json::Value   jv;
      if (jr.parse(is, jv)) {
        if (jv["codes"].isArray()) {
          for (int i = 0; i < jv["codes"].size(); ++i) {
            CountryCode  cc(jv["codes"][i]);
            if ((cc.Alpha2().length() == 2)
                && (cc.Alpha3().length() == 3)
                && (! cc.Name().empty())) {
              _a2toCountryCode[cc.Alpha2()] = cc;
              _a3toCountryCode[cc.Alpha3()] = cc;
            }
          }
        }
      }
      is.close();
    }
    return (! _a2toCountryCode.empty());
  }
  
  //--------------------------------------------------------------------------
  //!  
  //--------------------------------------------------------------------------
  bool CountryCodes::WriteJson(const string & filename) const
  {
    bool  rc = false;
    ofstream  os(filename.c_str());
    if (os) {
      Json::Value  jv;
      jv["description"] = string("ISO country codes");
      int  i = 0;
      for (auto & cci : _a2toCountryCode) {
        jv["codes"][i] = cci.second.Json();
        ++i;
      }
      os << Json::StyledWriter().write(jv);
      if (os) {
        rc = true;
      }
      os.close();
    }
    return rc;
  }

  //--------------------------------------------------------------------------
  //!  
  //--------------------------------------------------------------------------
  istream & CountryCodes::Read(istream & is)
  {
    _a3toCountryCode.clear();
    IO::Read(is, _a2toCountryCode);
    if (! _a2toCountryCode.empty()) {
      for (auto & cc : _a2toCountryCode) {
        _a3toCountryCode[cc.second.Alpha3()] = cc.second;
      }
    }
    return is;
  }

  //--------------------------------------------------------------------------
  //!  
  //--------------------------------------------------------------------------
  ssize_t CountryCodes::Read(int fd)
  {
    _a3toCountryCode.clear();
    ssize_t  rc = IO::Read(fd, _a2toCountryCode);
    if (! _a2toCountryCode.empty()) {
      for (auto & cc : _a2toCountryCode) {
        _a3toCountryCode[cc.second.Alpha3()] = cc.second;
      }
    }
    return rc;
  }

  //--------------------------------------------------------------------------
  //!  
  //--------------------------------------------------------------------------
  size_t CountryCodes::Read(FILE *f)
  {
    _a3toCountryCode.clear();
    size_t  rc = IO::Read(f, _a2toCountryCode);
    if (! _a2toCountryCode.empty()) {
      for (auto & cc : _a2toCountryCode) {
        _a3toCountryCode[cc.second.Alpha3()] = cc.second;
      }
    }
    return rc;
  }

  //--------------------------------------------------------------------------
  //!  
  //--------------------------------------------------------------------------
  ostream & CountryCodes::Write(ostream & os) const
  {
    return IO::Write(os, _a2toCountryCode);
  }

  //--------------------------------------------------------------------------
  //!  
  //--------------------------------------------------------------------------
  ssize_t CountryCodes::Write(int fd) const
  {
    return IO::Write(fd, _a2toCountryCode);
  }

  //--------------------------------------------------------------------------
  //!  
  //--------------------------------------------------------------------------
  size_t CountryCodes::Write(FILE *f) const
  {
    return IO::Write(f, _a2toCountryCode);
  }
  
  //--------------------------------------------------------------------------
  //!  
  //--------------------------------------------------------------------------
  uint32_t CountryCodes::StreamedLength() const
  {
    return IO::StreamedLength(_a2toCountryCode);
  }
  

}  // namespace Dwm

