//===========================================================================
// @(#) $Name:$
// @(#) $Id: DwmRDAPBootstrap.cc 9306 2017-05-08 09:04:02Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2017
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmRDAPBootstrap.cc
//!  \brief Dwm::RDAP::Bootstram implementation
//---------------------------------------------------------------------------

#include <fstream>
#include <iostream>
#include <iomanip>
#include <string>

#include <Poco/Net/PrivateKeyPassphraseHandler.h>
#include <Poco/Net/HTTPClientSession.h>
#include <Poco/Net/HTTPSClientSession.h>
#include <Poco/Net/HTTPRequest.h>
#include <Poco/Net/HTTPResponse.h>
#include <Poco/Net/SSLManager.h>
#include <Poco/Net/ConsoleCertificateHandler.h>
#include <Poco/URI.h>

#include "DwmSvnTag.hh"
#include "DwmRDAPBootstrap.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/libDwmRDAP/tags/libDwmRDAP-0.2.1/src/DwmRDAPBootstrap.cc 9306 $");

using namespace std;
using Poco::URI;
using Poco::Net::HTTPClientSession;
using Poco::Net::HTTPSClientSession;
using Poco::Net::HTTPMessage;
using Poco::Net::HTTPRequest;
using Poco::Net::HTTPResponse;
using Poco::Net::Context;

namespace Dwm {

  namespace RDAP {

    static Context::Ptr  g_context =
      new Context(Context::CLIENT_USE, "", "", "",
                  Context::VERIFY_NONE, 9, false,
                  "ALL:!ADH:!LOW:!EXP:!MD5:@STRENGTH");

    std::mutex                             Bootstrap::_mtx;
    Ipv4Routes<std::vector<std::string> >  Bootstrap::_addrs2rdaps;

    //------------------------------------------------------------------------
    //!  This is only here to force a default load of the bootstrap.  There
    //!  are no non-static members of the Bootstrap class.
    //------------------------------------------------------------------------
    static Bootstrap  g_bootstrap;

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    Bootstrap::Bootstrap()
    {
      if (Empty()) {
        Load();
      }
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Bootstrap::Load(const std::string & bootstrapURI)
    {
      {
        lock_guard<mutex>  lock(_mtx);
        _addrs2rdaps.Clear();
        Json::Value  &&jv = GetJsonResponse(bootstrapURI);
        for (int s = 0; s < jv["services"].size(); ++s) {
          for (int n = 0; n < jv["services"][s][0].size(); ++n) {
            for (int rds = 0; rds < jv["services"][s][1].size(); ++rds) {
              vector<string> & e =
                _addrs2rdaps[Ipv4Prefix(jv["services"][s][0][n].asString())];
              e.push_back(jv["services"][s][1][rds].asString());
              sort(e.begin(), e.end());
            }
          }
        }
      }
      
      return (! _addrs2rdaps.Empty());
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Bootstrap::Empty()
    {
      lock_guard<mutex>  lock(_mtx);
      return _addrs2rdaps.Empty();
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    vector<string> Bootstrap::RDAPServers(const Ipv4Address & addr)
    {
      vector<string>                   rc;
      pair<Ipv4Prefix,vector<string>>  result;
      lock_guard<mutex>                lock(_mtx);
      if (_addrs2rdaps.FindLongest(addr, result)) {
        rc = result.second;
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    Ipv4Prefix Bootstrap::MatchingPrefix(const Ipv4Address & addr)
    {
      Ipv4Prefix  rc("0/0");
      pair<Ipv4Prefix,vector<string>>  result;
      lock_guard<mutex>                lock(_mtx);
      if (_addrs2rdaps.FindLongest(addr, result)) {
        rc = result.first;
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    Json::Value Bootstrap::GetJsonResponse(const std::string & uristr)
    {
      Poco::URI    uri(uristr);
      Json::Value  rc;
      std::string  host(uri.getHost());
      std::string  path(uri.getPathAndQuery());
      HTTPClientSession    *sess;
      if (uri.getScheme() == "file") {
        ifstream  is(path.c_str());
        if (is) {
          Json::Reader  reader;
          reader.parse(is, rc);
        }
        return rc;
      }
      else if (uri.getScheme() == "https") {
        sess = new HTTPSClientSession(host, uri.getPort(), g_context);
      }
      else {
        sess = new HTTPClientSession(host);
      }
      HTTPRequest  req(HTTPRequest::HTTP_GET, path, HTTPMessage::HTTP_1_1);
      try {
        ostream & os = sess->sendRequest(req);
        if (os) {
          HTTPResponse  resp;
          istream  & is = sess->receiveResponse(resp);
          Json::Reader  reader;
          reader.parse(is, rc);
        }
      }
      catch (Poco::Exception &ex) {
        cerr << "Exception in sess.receiveResponse, URI "
             << uri.toString() << ", " << ex.className() << ", "
             << ex.name() << ", " << ex.displayText() << "\n";
      }
      delete sess;
      return rc;
    }

  }  // namespace RDAP

}  // namespace Dwm
