//===========================================================================
// @(#) $Name:$
// @(#) $Id: DwmRDAPFetcher.cc 9458 2017-06-06 05:08:27Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2017
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmRDAPFetcher.cc
//!  \brief NOT YET DOCUMENTED
//---------------------------------------------------------------------------

extern "C" {
  #include <sys/types.h>
  #include <sys/socket.h>
  #include <netinet/in.h>
  #include <netinet/tcp.h>
  #include <arpa/inet.h>
  #include <netdb.h>
}

#include "DwmDateTime.hh"
#include "DwmSvnTag.hh"
#include "DwmSysLogger.hh"
#include "DwmAuthPeerAuthenticator.hh"
#include "DwmRDAPFetcher.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/libDwmRDAP/tags/libDwmRDAP-0.2.1/src/DwmRDAPFetcher.cc 9458 $");

using namespace std;

namespace Dwm {

  namespace RDAP {

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    FetchedEntry::FetchedEntry(const Ipv4Prefix & prefix,
                               const string & country,
                               const TimeValue64 & lastUpdated)
        : _prefix(prefix), _country(country), _lastUpdated(lastUpdated)
    {}
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const Ipv4Prefix & FetchedEntry::Prefix() const
    {
      return _prefix;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const Ipv4Prefix & FetchedEntry::Prefix(const Ipv4Prefix & prefix)
    {
      _prefix = prefix;
      return _prefix;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const string & FetchedEntry::Country() const
    {
      return _country;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const string & FetchedEntry::Country(const string & country)
    {
      _country = country;
      return _country;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const TimeValue64 & FetchedEntry::LastUpdated() const
    {
      return _lastUpdated;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const TimeValue64 &
    FetchedEntry::LastUpdated(const TimeValue64 & lastUpdated)
    {
      _lastUpdated = lastUpdated;
      return _lastUpdated;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    Fetcher::Fetcher()
        : _rdapSocket(), _agreedKey()
    {}

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    Fetcher::~Fetcher()
    {
      CloseSession();
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Fetcher::OpenSession(const string & rdapServer,
                              const string & privKeyPath,
                              const string & knownServicesPath)
    {
      bool  rc = false;
      
      if (! rdapServer.empty()) {
        struct hostent  *hostEntry =
          gethostbyname2(rdapServer.c_str(), AF_INET);
        if (hostEntry) {
          struct sockaddr_in  servAddr;
          memset(&servAddr, 0, sizeof(servAddr));
#ifndef __linux__
          servAddr.sin_len = sizeof(servAddr);
#endif
          servAddr.sin_family = AF_INET;
          servAddr.sin_port = htons(6363);
          servAddr.sin_addr.s_addr = *(in_addr_t *)hostEntry->h_addr_list[0];
          if (_rdapSocket.Open(PF_INET, SOCK_STREAM, 0)) {
            if (_rdapSocket.Connect(servAddr)) {
              Auth::PeerAuthenticator  peer(privKeyPath, knownServicesPath);
              string  theirId;
              if (peer.Authenticate(_rdapSocket, theirId, _agreedKey)) {
                rc = true;
              }
              else {
                _rdapSocket.Close();
                Syslog(LOG_ERR, "Failed to authenticate to dwmrdapd@%s",
                       rdapServer.c_str());
              }
            }
            else {
              _rdapSocket.Close();
              Syslog(LOG_ERR, "Failed to connect to dwmrdapd@%s",
                     rdapServer.c_str());
            }
          }
          else {
            Syslog(LOG_ERR, "Failed to open socket for dwmrdapd@%s",
                   rdapServer.c_str());
          }
        }
        else {
          Syslog(LOG_ERR, "Unknown dwmrdapd host %s", rdapServer.c_str());
        }
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Fetcher::RequestResponse(const RDAP::RequestMessage & request,
                                  RDAP::ResponseMessage & response)
    {
      bool  rc = false;
      if (_rdapSocket) {
        if (request.Write(_rdapSocket, _agreedKey)) {
          if (response.Read(_rdapSocket, _agreedKey)) {
            rc = true;
          }
        }
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Fetcher::ResponseEntryHasData(const Json::Value & jv)
    {
      return (jv.isMember("ipv4addr")
              && jv["ipv4addr"].isString()
              && jv.isMember("prefix")
              && jv["prefix"].isString()
              && jv.isMember("country")
              && jv["country"].isString()
              && jv.isMember("lastUpdated")
              && jv["lastUpdated"].isString());
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    FetchedEntryMap Fetcher::GetEntries(const vector<Ipv4Address> & addrs)
    {
      FetchedEntryMap  entryMap;
      if (_rdapSocket) {
        RDAP::RequestMessage          rdapReq(addrs);
        RDAP::ResponseMessage         rdapResp;
        if (RequestResponse(rdapReq, rdapResp)) {
          const Json::Value  & jv = rdapResp.Json();
          if ((! jv.empty()) && jv.isArray() && (jv.size() > 0)) {
            DateTime  dt;
            for (int j = 0; j < jv.size(); ++j) {
              if (ResponseEntryHasData(jv[j])) {
                Ipv4Address addr(jv[j]["ipv4addr"].asString());
                Ipv4Prefix  pfx(jv[j]["prefix"].asString());
                dt.Parse("%Y-%m-%d %H:%M", jv[j]["lastUpdated"].asString());
                FetchedEntry  fe(pfx, jv[j]["country"].asString(),
                                 dt.GetTimeValue64());
                entryMap[addr] = fe;
              }
            }
          }
        }
      }
      return entryMap;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    pair<bool,FetchedEntry> Fetcher::GetEntry(const Ipv4Address & addr)
    {
      pair<bool,FetchedEntry>  rc;
      rc.first = false;
      vector<Ipv4Address>  addrs = { addr };
      FetchedEntryMap  &&entries = GetEntries(addrs);
      if (! entries.empty()) {
        rc.first = true;
        rc.second = entries.begin()->second;
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void Fetcher::CloseSession()
    {
      _rdapSocket.Close();
      _agreedKey.assign(_agreedKey.size(), '\0');
      return;
    }
    
    
    
  }  // namespace RDAP

}  // namespace Dwm
