//===========================================================================
// @(#) $Name:$
// @(#) $Id: DwmRDAPResponder.hh 12106 2022-12-04 06:18:56Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2017
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmRDAPResponder.hh
//!  \brief Dwm::RDAP::Responder class definition
//---------------------------------------------------------------------------

#ifndef _DWMRDAPRESPONDER_HH_
#define _DWMRDAPRESPONDER_HH_


#include <string>
#include <thread>

#include "DwmCredencePeer.hh"
#include "DwmIpv4CountryDb.hh"
#include "DwmCountryCodes.hh"
#include "DwmRDAPBootstrap.hh"
#include "DwmRDAPQuerySessions.hh"

namespace Dwm {

  namespace RDAP {

    class Server;
    
    //------------------------------------------------------------------------
    //!  Encapsulates a thread that responds to request from a single
    //!  client.
    //------------------------------------------------------------------------
    class Responder
    {
    public:
      //----------------------------------------------------------------------
      //!  Construct from socket, path to my RSA private key file,
      //!  path to an authorized_keys file, the database and country codes.
      //----------------------------------------------------------------------
      Responder(boost::asio::ip::tcp::socket && s, Server & server,
                const CountryCodes & cc);

      //----------------------------------------------------------------------
      //!  Destructor.
      //----------------------------------------------------------------------
      ~Responder();

      //----------------------------------------------------------------------
      //!  Joins the responder thread.
      //----------------------------------------------------------------------
      bool Join();
      
    private:
      Credence::Peer       _peer;
      Server              &_server;
      Ipv4CountryDb       &_db;
      const CountryCodes  &_cc;
      QuerySessions        _querySessions;
      std::thread          _thread;
      std::atomic<bool>    _running;
      
      bool GetDbResult(const Ipv4Address & addr,
                       std::pair<Ipv4Prefix,Ipv4CountryDbValue> & match) const;
      
      bool IsValidRequest(const nlohmann::json & request) const;
      bool IsValidQueryRequest(const nlohmann::json & request) const;
      bool IsValidDeleteRequest(const nlohmann::json & request) const;
      
      nlohmann::json
      DbMatchToJson(const Ipv4Address & addr,
                    const std::pair<Ipv4Prefix,Ipv4CountryDbValue> & match) const;
      static nlohmann::json MulticastToJson(const Ipv4Address & addr);
      static nlohmann::json PrivateToJson(const Ipv4Address & addr);
      static nlohmann::json FutureToJson(const Ipv4Address & addr);
      static bool IsSpecialAddress(const Ipv4Address & addr,
                                   nlohmann::json & jv);
      nlohmann::json HandleRequest(const nlohmann::json & request,
                                   bool & dbupdated);
      nlohmann::json HandleQueryRequest(const nlohmann::json & request,
                                        bool & dbupdated);
      nlohmann::json HandleDeleteRequest(const nlohmann::json & request,
                                         bool & dbupdated);
      void Run();
    };
      
  }  // namespace RDAP

}  // namespace Dwm

#endif  // _DWMRDAPRESPONDER_HH_
