//===========================================================================
// @(#) $Name:$
// @(#) $Id: dwmrdapc.cc 12126 2022-12-11 04:47:28Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2017
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file dwmrdapc.cc
//!  \brief dwmrdap client, mostly just an example
//---------------------------------------------------------------------------

#include <cstdlib>
#include <fstream>
#include <regex>
#include <sstream>
#include <string>

#include "DwmIpv4Prefix.hh"
#include "DwmOptArgs.hh"
#include "DwmSvnTag.hh"
#include "DwmSysLogger.hh"
#include "DwmCredencePeer.hh"
#include "DwmRDAPRequestMessage.hh"
#include "DwmRDAPResponseMessage.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/libDwmRDAP/tags/libDwmRDAP-0.3.4/apps/dwmrdapc/dwmrdapc.cc 12126 $");

using namespace std;
using namespace Dwm;

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
string RequestResponse(Credence::Peer & peer,
                       const RDAP::RequestMessage & request)
{
  string                  rc;
  RDAP::ResponseMessage   response;
  if (request.Write(peer)) {
    if (response.Read(peer)) {
      rc = response.Json().dump(2);
    }
    else {
      Syslog(LOG_ERR, "Failed to read server response");
    }
  }
  else {
    Syslog(LOG_ERR, "Failed to write server request");
  }
  return rc;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
bool RequestResponse(Credence::Peer & peer,
                     const RDAP::RequestMessage & request,
                     RDAP::ResponseMessage & response)
{
  bool  rc = false;
  if (request.Write(peer)) {
    if (response.Read(peer)) {
      rc = true;
    }
    else {
      Syslog(LOG_ERR, "Failed to read server response");
    }
  }
  else {
    Syslog(LOG_ERR, "Failed to write server request");
  }
  return rc;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static void Usage(const string & argv0)
{
  cerr << "Usage: " << argv0 << " [-h host] IPv4_address\n";
  return;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static bool Query(Credence::Peer & peer,
                  const vector<Ipv4Address> & addrs)
{
  bool  rc = false;
  RDAP::RequestMessage  req(addrs);
  string  resp = RequestResponse(peer, req);
  if (! resp.empty()) {
    cout << resp << '\n';
    rc = true;
  }
  return rc;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static bool Delete(Credence::Peer & peer,
                   const vector<Ipv4Prefix> & prefixes)
{
  bool  rc = false;
  return rc;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
Ipv4Address NextWalkAddress(const Ipv4Prefix & prev, Ipv4Prefix & resp)
{
  Ipv4Address  addr;
  if (resp.MaskLength() < 9) {
    addr = prev.FirstAddress();
    --addr;
    resp = Ipv4Prefix(addr, 24);
  }
  else {
    addr = resp.FirstAddress();
    --addr;
  }
  return addr;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static void Walk(Credence::Peer & peer, const Ipv4Prefix & pfx)
{
  vector<Ipv4Address>  addrs;
  addrs.push_back(pfx.LastAddress());
  Ipv4Prefix  nextPfx(pfx);
  while (addrs[0] >= pfx.FirstAddress()) {
    RDAP::RequestMessage   req(addrs);
    RDAP::ResponseMessage  resp;
    bool  rc = RequestResponse(peer, req, resp);
    if (rc) {
      if ((resp.Json()[0].find("prefix") != resp.Json()[0].end())
          && resp.Json()[0]["prefix"].is_string()) {
        Ipv4Prefix  respPfx =
          Ipv4Prefix(resp.Json()[0]["prefix"].get<string>());
        addrs[0] = NextWalkAddress(nextPfx, respPfx);
        cout << resp.Json().dump(2) << '\n';
        nextPfx = respPfx;
      }
      else {
        addrs[0] -= 256;
        nextPfx = Ipv4Prefix(addrs[0], 24);
        // addrs[0] = nextPfx.Network();
        cout << "pfx, addr from no response: " << nextPfx.ToShortString()
             << ' ' << addrs[0] << '\n';
      }
    }
    else {
      break;
    }
  }
  return;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
int main(int argc, char *argv[])
{
  int      rc = 1;
  OptArgs  optargs;
  optargs.AddOptArg("d", "delete", false, "", "delete prefixes");
  optargs.AddOptArg("h:", "host", false, "", "server host");
  optargs.AddOptArg("w", "prefix", false, "", "walk a prefix");
  
  int  nextArg = optargs.Parse(argc, argv);
#if 0
  if (nextArg >= argc - 1) {
    Usage(argv[0]);
    exit(1);
  }
#endif
  
  SysLogger::Open("dwmrdapc", LOG_PERROR, LOG_USER);
  SysLogger::MinimumPriority("info");

  string  rdapHost = optargs.Get<string>('h');
  if (rdapHost.empty()) {
    if (getenv("RDAPDHOST")) {
      rdapHost = getenv("RDAPDHOST");
    }
  }
  if (rdapHost.empty()) {
    cerr << "A host is required.  Either set RDAPDHOST in your environment\n"
         << "or use the '-h host' command line option.\n\n";
    Usage(argv[0]);
    exit(1);
  }

  Credence::Peer  peer;
  if (peer.Connect(rdapHost, 6363)) {
    Credence::KeyStash   keyStash;
    Credence::KnownKeys  knownKeys;
    if (! peer.Authenticate(keyStash, knownKeys)) {
      cerr << "Authentication failed\n";
      exit(1);
    }
  }
  else {
    cerr << "Failed to connect to " << rdapHost << '\n';
    exit(1);
  }
  
  if (argc > nextArg) {
    if (optargs.Get<bool>('d')) {
      vector<Ipv4Prefix>  prefixes;
      for ( ; nextArg < argc; ++nextArg) {
        prefixes.push_back(Ipv4Prefix(argv[nextArg]));
      }
      if (Delete(peer, prefixes)) {
        rc = 0;
      }
    }
    else if (optargs.Get<bool>('w')) {
      for ( ; nextArg < argc; ++nextArg) {
        Walk(peer, Ipv4Prefix(argv[nextArg]));
      }
    }
    else {
      vector<Ipv4Address>  addrs;
      for ( ; nextArg < argc; ++nextArg) {
        addrs.push_back(Ipv4Address(argv[nextArg]));
      }
      if (Query(peer, addrs)) {
        rc = 0;
      }
    }
  }
  else {
    Usage(argv[0]);
    exit(1);
  }

  return rc;
}
