//===========================================================================
// @(#) $Name:$
// @(#) $Id: DwmRDAPResponder.cc 12124 2022-12-11 04:45:09Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2017, 2022
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmRDAPResponder.cc
//!  \brief Dwm::RDAP::Responder class implementation
//---------------------------------------------------------------------------

extern "C" {
  #include <sys/types.h>
  #include <sys/socket.h>
  #include <netinet/in.h>
  #include <netinet/tcp.h>
  #include <arpa/inet.h>
#ifndef __APPLE__
  #ifndef __linux__
    #include <pthread_np.h>
  #else
    #include <pthread.h>
  #endif
#endif
}

#include "DwmSysLogger.hh"
#include "DwmSvnTag.hh"
#include "libDwmRDAPPortability.hh"
#include "DwmRDAPQuery.hh"
#include "DwmRDAPRequestMessage.hh"
#include "DwmRDAPResponseMessage.hh"
#include "DwmRDAPServer.hh"

static const Dwm::SvnTag  svntag("$DwmPath: dwm/libDwmRDAP/tags/libDwmRDAP-0.3.4/apps/dwmrdapd/DwmRDAPResponder.cc 12124 $");

using namespace std;

namespace Dwm {

  namespace RDAP {

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    Responder::Responder(boost::asio::ip::tcp::socket && s, Server & server,
                         const CountryCodes & cc)
        : _peer(), _server(server), _db(server.GetDb()), _cc(cc),
          _querySessions(), _running(false)
    {
      boost::system::error_code  ec;
      s.native_non_blocking(false, ec);
      _peer.SetKeyExchangeTimeout(chrono::milliseconds(2000));
      _peer.SetIdExchangeTimeout(chrono::milliseconds(2000));
      if (_peer.Accept(std::move(s))) {
        _thread = thread(&Responder::Run, this);
        _running.store(true);
      }
      else {
        _peer.Disconnect();
        Syslog(LOG_ERR, "Failed to accept peer");
      }
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    Responder::~Responder()
    {
      Join();
      Syslog(LOG_INFO, "Responder destroyed for %s", _peer.Id().c_str());
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Responder::Join()
    {
      bool  rc = false;
      bool  isRunning = _running.load();
      if (! isRunning) {
        if (_thread.joinable()) {
          _thread.join();
          rc = true;
          Syslog(LOG_DEBUG, "Joined thread for responder %s",
                 _peer.Id().c_str());
        }
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Responder::IsValidQueryRequest(const nlohmann::json & request) const
    {
      bool  rc = false;
      if (request.find("type") != request.end()) {
        if (request["type"].is_string()
            && (request["type"].get<string>() == "query")) {
          if ((request.find("ipv4addrs") != request.end())
              && request["ipv4addrs"].is_array()) {
            int  i = 0;
            for ( ; i < request["ipv4addrs"].size(); ++i) {
              if (! request["ipv4addrs"][i].is_string()) {
                break;
              }
              struct in_addr  inAddr;
              if (inet_pton(AF_INET,
                            request["ipv4addrs"][i].get<string>().c_str(),
                            &inAddr) != 1) {
                break;
              }
            }
            if (i == request["ipv4addrs"].size()) {
              rc = true;
            }
          }
        }
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Responder::IsValidDeleteRequest(const nlohmann::json & request) const
    {
      bool  rc = false;
      if (request.find("type") != request.end()) {
        if (request["type"].is_string()
            && (request["type"].get<string>() == "delete")) {
          if ((request.find("prefixes") != request.end())
              && request["prefixes"].is_array()) {
            int  i = 0;
            for ( ; i < request["prefixes"].size(); ++i) {
              if (! request["prefixes"][i].is_string()) {
                break;
              }
              struct in_addr  inAddr;
              if (inet_net_pton(AF_INET,
                                request["prefixes"][i].get<string>().c_str(),
                                &inAddr, sizeof(inAddr)) == -1) {
                break;
              }
            }
            if (i == request["prefixes"].size()) {
              rc = true;
            }
          }
        }
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Responder::IsValidRequest(const nlohmann::json & request) const
    {
      return (IsValidQueryRequest(request)
              || IsValidDeleteRequest(request));
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    nlohmann::json
    Responder::DbMatchToJson(const Ipv4Address & addr,
                             const pair<Ipv4Prefix,Ipv4CountryDbValue> & match) const
    {
      nlohmann::json  jv = match.second.Json();
      jv["ipv4addr"] = (string)addr;
      jv["countryName"] = _cc.FindByCode(match.second.Code()).Name();
      jv["prefix"] = match.first.ToShortString();
      return jv;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    nlohmann::json Responder::MulticastToJson(const Ipv4Address & addr)
    {
      nlohmann::json  jv;
      DateTime  now(TimeValue64(true));
      jv["ipv4addr"] = (string)addr;
      jv["country"] = "22";
      jv["countryName"] = "multicast";
      jv["lastUpdated"] = now.Formatted("%Y-%m-%d %H:%M");
      jv["lastChanged"] = now.Formatted("%Y-%m-%d %H:%M");
      jv["prefix"] = "224/4";
      return jv;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    nlohmann::json Responder::PrivateToJson(const Ipv4Address & addr)
    {
      nlohmann::json  jv;
      DateTime  now(TimeValue64(true));
      jv["ipv4addr"] = (string)addr;
      jv["country"] = "10";
      jv["countryName"] = "private";
      jv["lastUpdated"] = now.Formatted("%Y-%m-%d %H:%M");
      jv["lastChanged"] = now.Formatted("%Y-%m-%d %H:%M");
      jv["prefix"] = Bootstrap::MatchingPrefix(addr).ToShortString();
      return jv;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    nlohmann::json Responder::FutureToJson(const Ipv4Address & addr)
    {
      nlohmann::json  jv;
      DateTime  now(TimeValue64(true));
      jv["ipv4addr"] = (string)addr;
      jv["country"] = "24";
      jv["countryName"] = "future";
      jv["lastUpdated"] = now.Formatted("%Y-%m-%d %H:%M");
      jv["lastChanged"] = now.Formatted("%Y-%m-%d %H:%M");
      jv["prefix"] = "240/4";
      return jv;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool
    Responder::GetDbResult(const Ipv4Address & addr,
                           pair<Ipv4Prefix,Ipv4CountryDbValue> & match) const
    {
      return _db.FindLongest(addr, match);
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Responder::IsSpecialAddress(const Ipv4Address & addr,
                                     nlohmann::json & jv)
    {
      bool  rc = false;
      vector<string> rdapServers = Bootstrap::RDAPServers(addr);
      if (rdapServers.size() == 1) {
        const string & rds = rdapServers.front();
        if (rds == "multicast") {
          jv = MulticastToJson(addr);
          rc = true;
        }
        else if (rds == "private") {
          jv = PrivateToJson(addr);
          rc = true;
        }
        else if (rds == "future") {
          jv = FutureToJson(addr);
          rc = true;
        }
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    nlohmann::json
    Responder::HandleDeleteRequest(const nlohmann::json & request,
                                   bool & dbupdated)
    {
      nlohmann::json  rc;
      dbupdated = false;
      Ipv4CountryDbValue  match;
      for (int i = 0; i < request["prefixes"].size(); ++i) {
        if (request["prefixes"][i].is_string()) {
          Ipv4Prefix  pfx(request["prefixes"][i].get<string>());
          if (! _db.Find(pfx, match)) {
            rc["prefixes"][i] = string(pfx.ToShortString() + " not found");
          }
          else {
            _db.Delete(pfx);
            rc["prefixes"][i] =	string(pfx.ToShortString() + " deleted");
            dbupdated = true;
          }
        }
        else {
          rc["prefixes"][i] = string("invalid prefix\n");
        }
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    nlohmann::json
    Responder::HandleQueryRequest(const nlohmann::json & request,
                                  bool & dbupdated)
    {
      nlohmann::json  rc;
      vector<Ipv4Address>  addrs;
      for (int i = 0; i < request["ipv4addrs"].size(); ++i) {
        if (request["ipv4addrs"][i].is_string()) {
          addrs.push_back(Ipv4Address(request["ipv4addrs"][i].get<string>()));
        }
      }
      TimeValue64  expireTime(true);
      expireTime -= TimeValue64(90 * 24 * 60 * 60, 0);
      for (int i = 0; i < addrs.size(); ++i) {
        nlohmann::json  dbVal;
        pair<Ipv4Prefix,Ipv4CountryDbValue>  match;
        if (IsSpecialAddress(addrs[i], rc[i])) {
        }
        else if (GetDbResult(addrs[i], match)
                 && (match.second.LastUpdated() > expireTime)) {
          rc[i] = DbMatchToJson(addrs[i], match);
        }
        else {
          Syslog(LOG_DEBUG, "Making RDAP query for %s",
                 ((string)addrs[i]).c_str());
          RDAP::IPv4Query     query(addrs[i]);
          RDAP::IPv4Response  rdapResult = _querySessions.Execute(query);
          if (_db.Update(rdapResult, 90)) {
            dbupdated = true;
          }
          else {
            Syslog(LOG_ERR, "Failed to update database for %s",
                   ((string)addrs[i]).c_str());
          }
          if (GetDbResult(addrs[i], match)) {
            rc[i] = DbMatchToJson(addrs[i], match);
          }
          else {
          }
        }
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    nlohmann::json Responder::HandleRequest(const nlohmann::json & request,
                                            bool & dbupdated)
    {
      nlohmann::json       rc;
      if (IsValidQueryRequest(request)) {
        rc = HandleQueryRequest(request, dbupdated);
      }
      else if (IsValidDeleteRequest(request)) {
        rc = HandleDeleteRequest(request, dbupdated);
      }
      else {
        if (SysLogger::MinimumPriority() >= LOG_DEBUG) {
          Syslog(LOG_DEBUG, "invalid request %s", request.dump().c_str());
        }
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void Responder::Run()
    {
      set_pthread_name(_thread.native_handle(), "responder");
      RequestMessage    reqMsg;
      bool              savedb = false;

      Syslog(LOG_INFO, "Responder started");
      if (_peer.Authenticate(_server.GetKeyStash(), _server.GetKnownKeys())) {
        while (reqMsg.Read(_peer)) {
          bool  dbupdated = false;
          const nlohmann::json  & reqJson = reqMsg.Json();
          nlohmann::json  respJson = HandleRequest(reqJson, dbupdated);
          if (dbupdated) {
            savedb = true;
          }
          ResponseMessage respMsg(respJson);
          if (! respMsg.Write(_peer)) {
            break;
          }
        }
        if (savedb) {
          _db.Save();
        }
      }
      _running = false;
      Syslog(LOG_INFO, "Responder done");
      return;
    }

  }  // namespace RDAP

}  // namespace Dwm
