//===========================================================================
// @(#) $Name:$
// @(#) $Id: DwmRDAPFetcher.cc 12123 2022-12-11 04:44:23Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2017
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmRDAPFetcher.cc
//!  \brief Dwm::RDAP::Fetcher class implementation
//---------------------------------------------------------------------------

#include "DwmDateTime.hh"
#include "DwmSvnTag.hh"
#include "DwmSysLogger.hh"
#include "DwmRDAPFetcher.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/libDwmRDAP/tags/libDwmRDAP-0.3.4/src/DwmRDAPFetcher.cc 12123 $");

using namespace std;

namespace Dwm {

  namespace RDAP {

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    FetchedEntry::FetchedEntry(const Ipv4Prefix & prefix,
                               const string & country,
                               const TimeValue64 & lastUpdated)
        : _prefix(prefix), _country(country), _lastUpdated(lastUpdated)
    {}
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const Ipv4Prefix & FetchedEntry::Prefix() const
    {
      return _prefix;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const Ipv4Prefix & FetchedEntry::Prefix(const Ipv4Prefix & prefix)
    {
      _prefix = prefix;
      return _prefix;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const string & FetchedEntry::Country() const
    {
      return _country;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const string & FetchedEntry::Country(const string & country)
    {
      _country = country;
      return _country;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const TimeValue64 & FetchedEntry::LastUpdated() const
    {
      return _lastUpdated;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const TimeValue64 &
    FetchedEntry::LastUpdated(const TimeValue64 & lastUpdated)
    {
      _lastUpdated = lastUpdated;
      return _lastUpdated;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    Fetcher::Fetcher()
    {}

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    Fetcher::~Fetcher()
    {
      CloseSession();
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Fetcher::OpenSession(const string & rdapServer,
                              const Credence::KeyStash & keyStash,
                              const Credence::KnownKeys & knownKeys)
    {
      bool  rc = false;
      
      if (! rdapServer.empty()) {
        if (_peer.Connect(rdapServer, 6363)) {
          if (_peer.Authenticate(keyStash, knownKeys)) {
            rc = true;
          }
          else {
            Syslog(LOG_ERR, "Authentication to %s failed",
                   rdapServer.c_str());
          }
        }
        else {
          Syslog(LOG_ERR, "Failed to connect to %s", rdapServer.c_str());
        }
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Fetcher::RequestResponse(const RDAP::RequestMessage & request,
                                  RDAP::ResponseMessage & response)
    {
      bool  rc = false;
      if (request.Write(_peer)) {
        if (response.Read(_peer)) {
          rc = true;
        }
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Fetcher::ResponseEntryHasData(const nlohmann::json & jv)
    {
      return ((jv.find("ipv4addr") != jv.end())
              && jv["ipv4addr"].is_string()
              && (jv.find("prefix") != jv.end())
              && jv["prefix"].is_string()
              && (jv.find("country") != jv.end())
              && jv["country"].is_string()
              && (jv.find("lastUpdated") != jv.end())
              && jv["lastUpdated"].is_string());
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    FetchedEntryMap Fetcher::GetEntries(const vector<Ipv4Address> & addrs)
    {
      FetchedEntryMap        entryMap;
      RDAP::RequestMessage   rdapReq(addrs);
      RDAP::ResponseMessage  rdapResp;
      if (RequestResponse(rdapReq, rdapResp)) {
        const nlohmann::json  & jv = rdapResp.Json();
        if ((! jv.empty()) && jv.is_array() && (jv.size() > 0)) {
          DateTime  dt;
          for (int j = 0; j < jv.size(); ++j) {
            if (ResponseEntryHasData(jv[j])) {
              Ipv4Address addr(jv[j]["ipv4addr"].get<string>());
              Ipv4Prefix  pfx(jv[j]["prefix"].get<string>());
              dt.Parse("%Y-%m-%d %H:%M", jv[j]["lastUpdated"].get<string>());
              FetchedEntry  fe(pfx, jv[j]["country"].get<string>(),
                               dt.GetTimeValue64());
              entryMap[addr] = fe;
            }
          }
        }
      }
      return entryMap;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    pair<bool,FetchedEntry> Fetcher::GetEntry(const Ipv4Address & addr)
    {
      pair<bool,FetchedEntry>  rc;
      rc.first = false;
      vector<Ipv4Address>  addrs = { addr };
      FetchedEntryMap  entries = GetEntries(addrs);
      if (! entries.empty()) {
        rc.first = true;
        rc.second = entries.begin()->second;
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void Fetcher::CloseSession()
    {
      _peer.Disconnect();
      return;
    }
    
  }  // namespace RDAP

}  // namespace Dwm
