//===========================================================================
// @(#) $Name:$
// @(#) $Id: mcblockc.cc 12101 2022-12-04 01:17:28Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2017
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file mcblockc.cc
//!  \brief mcblock command-line client
//---------------------------------------------------------------------------

extern "C" {
  #include <sys/types.h>
  #include <sys/socket.h>
  #include <netinet/in.h>
  #include <netinet/tcp.h>
  #include <arpa/inet.h>
  #include <netdb.h>
  #include <pwd.h>
  #include <unistd.h>
  #include <fcntl.h>
}

#include <cassert>
#include <fstream>
#include <string>

#include "DwmCredencePeer.hh"
#include "DwmIO.hh"
#include "DwmIpv4Routes.hh"
#include "DwmOptArgs.hh"
#include "DwmSignal.hh"
#include "DwmSvnTag.hh"
#include "DwmSysLogger.hh"
#include "DwmMcBlockRequestMessage.hh"
#include "DwmMcBlockResponseMessage.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/mcplex/mcblock/tags/mcblock-0.3.3/apps/mcblockc/mcblockc.cc 12101 $");

using namespace std;
using namespace Dwm;

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static string GetHomeDirectory()
{
  string  rc;
  uid_t   uid = geteuid();
  struct passwd *pw = getpwuid(uid);
  if (pw) {
    rc = pw->pw_dir;
  }
  return rc;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static string GetKeyDirectory()
{
  string  rc;
  string  homeDir = GetHomeDirectory();
  if (! homeDir.empty()) {
    rc = homeDir + "/.credence";
  }
  return rc;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
bool RequestResponse(Credence::Peer & peer,
                     const McBlock::RequestMessage & request,
                     McBlock::ResponseMessage & response)
{
  bool  rc = false;
  if (request.Write(peer)) {
    if (response.Read(peer)) {
      rc = true;
    }
    else {
      Syslog(LOG_ERR, "Failed to read server response");
    }
  }
  else {
    Syslog(LOG_ERR, "Failed to write server request");
  }
  return rc;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
bool Connect(const string & host, Credence::Peer & peer)
{
  bool  rc = false;
  struct hostent  *hostEntry = gethostbyname(host.c_str());
  if (! hostEntry) {
    cerr << "Host '" << host << "' not found.\n";
    exit(1);
  }

  if (peer.Connect(host, 1001)) {
    rc = true;
  }
  return rc;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static void Usage(const string & argv0)
{
  cerr << "Usage: " << argv0 << " [-h host] activate tableName prefix(es)\n"
       << "       " << argv0 << " [-h host] deactivate tableName prefix(es)\n"
       << "       " << argv0 << " [-h host] getactive tableName\n"
       << "       " << argv0 << " [-h host] getaddrules tableName\n"
       << "       " << argv0 << " [-h host] loghit tableName ipv4addr\n"
       << "       " << argv0 << " [-h host] search ipv4addr\n"
       << "       " << argv0 << " [-h host] get tableName ipv4addr\n"
       << "       " << argv0 << " [-h host] edit tableName ipv4addr\n"
       << "       " << argv0 << " [-h host] ping\n";
  return;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static bool ActivatePrefixes(Credence::Peer & peer,const string & table,
                             const vector<Ipv4Prefix> & prefixes)
{
  bool  rc = false;
  McBlock::ResponseMessage  resp;
  McBlock::RequestMessage   req(McBlock::ActivateReq(table, prefixes));
  if (RequestResponse(peer, req, resp)) {
    if (! resp.Json().empty()) {
      cout << resp.Json().dump(2) << '\n';
      rc = true;
    }
  }
  return rc;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static bool DeactivatePrefixes(Credence::Peer & peer, const string & table,
                               const vector<Ipv4Prefix> & prefixes)
{
  bool  rc = false;
  McBlock::ResponseMessage  resp;
  McBlock::RequestMessage   req(McBlock::DeactivateReq(table, prefixes));
  if (RequestResponse(peer, req, resp)) {
    if (! resp.Json().empty()) {
      cout << resp.Json().dump(2) << '\n';
      rc = true;
    }
  }
  return rc;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static bool GetActivePrefixes(Credence::Peer & peer, const string & table)
{
  bool  rc = false;
  McBlock::ResponseMessage  resp;
  McBlock::RequestMessage   req((McBlock::GetActiveReq(table)));
  if (RequestResponse(peer, req, resp)) {
    if (! resp.Json().empty()) {
      if (resp.Json().is_object()) {
        for (auto & [key, value] : resp.Json().items()) {
          map<string,Ipv4Routes<uint8_t>>  countryRoutes;
          if ((resp.Json()[key].find("prefixes") != resp.Json()[key].end())
              && resp.Json()[key]["prefixes"].is_array()) {
            cout << key << ":\n";
            const nlohmann::json &  pfxArr = resp.Json()[key]["prefixes"];
            for (uint32_t pn = 0; pn < pfxArr.size(); ++pn) {
              Ipv4Prefix  prefix(pfxArr[pn]["prefix"].get<string>());
              string      country(pfxArr[pn]["country"].get<string>());
              cout << setiosflags(ios::left)
                   << "  "
                   << setw(18) << pfxArr[pn]["prefix"].get<string>() << ' '
                   << setiosflags(ios::right)
                   << setw(6) << pfxArr[pn]["daysRemaining"].get<uint32_t>() << "d "
                   << setw(3) << pfxArr[pn]["country"].get<string>() << " ("
                   << pfxArr[pn]["countryName"].get<string>() << ")\n"
                   << resetiosflags(ios::right) << resetiosflags(ios::left);
              countryRoutes[country][prefix] = 1;
            }
          }
          vector<pair<string,uint64_t>>  countryAddrSpace;
          for (auto & cre : countryRoutes) {
            countryAddrSpace.push_back(pair<string,uint64_t>(cre.first,
                                                             cre.second.AddressesCovered()));
          }
          sort (countryAddrSpace.begin(), countryAddrSpace.end(),
                [&] (const pair<string,uint64_t> & a1,
                     const pair<string,uint64_t> & a2)
                { return (a1.second > a2.second); });
          cout << "\n  Addresses covered per country:\n";
          std::cout.imbue(std::locale("en_US.UTF-8"));
          for (auto & cas : countryAddrSpace) {
            cout << "    " << cas.first << ' ' << cas.second << '\n';
            auto  cre = countryRoutes[cas.first];
            vector<pair<uint8_t,uint32_t>>  hmSizes;
            cre.HashSizes(hmSizes);
            for (auto hms : hmSizes) {
              cout << "      /" << setw(2) << (uint16_t)hms.first
                   << " networks: " << setw(4) << hms.second << " ("
                   << hms.second * (1UL << (32 - hms.first))
                   << " addresses)\n";
            }
          }
        }
      }
      rc = true;
    }
  }
  return rc;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static bool Search(Credence::Peer & peer, const Ipv4Address & addr)
{
  bool                      rc = false;
  McBlock::ResponseMessage  resp;
  McBlock::RequestMessage   req((McBlock::SearchReq(addr)));
  if (RequestResponse(peer, req, resp)) {
    if (! resp.Json().empty()) {
      if (resp.Json().is_object()) {
        // auto  && members = resp.Json().getMemberNames();
        for (auto & [key, value] : resp.Json().items()) {
          if ((resp.Json()[key].find("prefix") != resp.Json()[key].end())
              && resp.Json()[key]["prefix"].is_string()) {
            const nlohmann::json  & match = resp.Json()[key];
            cout << key << ":\n"
                 << setiosflags(ios::left)
                 << "  "
                 << setw(18) << match["prefix"].get<string>()
                 << ' ' << setiosflags(ios::right)
                 << setw(6) << match["daysRemaining"].get<uint32_t>() << "d "
                 << setw(3) << match["country"].get<string>() << " ("
                 << match["countryName"].get<string>() << ")\n"
                 << resetiosflags(ios::right) << resetiosflags(ios::left);
          }
          rc = true;
        }
      }
    }
  }
  return rc;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static bool Ping(Credence::Peer & peer)
{
  bool                      rc = false;
  McBlock::ResponseMessage  resp;
  nlohmann::json            jv;
  jv["type"] = "ping";
  McBlock::RequestMessage   req(jv);
  if (RequestResponse(peer, req, resp)) {
    if (! resp.Json().empty()) {
      if (resp.Json().find("alive") != resp.Json().end()) {
        if (resp.Json()["alive"].is_boolean()) {
          if (resp.Json()["alive"].get<bool>()) {
            cout << "alive\n";
            rc = true;
          }
        }
      }
    }
  }
  return rc;
}
  
//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static bool GetAddRules(Credence::Peer & peer, const string & table)
{
  bool  rc = false;
  McBlock::ResponseMessage  resp;
  McBlock::RequestMessage   req((McBlock::GetAddRulesReq(table)));
  if (RequestResponse(peer, req, resp)) {
    if (! resp.Json().empty()) {
      cout << resp.Json().dump(2) << '\n';
      rc = true;
    }
  }
  return rc;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static bool LogHit(Credence::Peer & peer, const string & table,
                   const string & addrstr)
{
  Dwm::TimeValue64  now(true);
  bool  rc = false;
  Ipv4Address  addr(addrstr);
  McBlock::ResponseMessage  resp;
  McBlock::RequestMessage   req((McBlock::LogHitReq(table, addr, now)));
  if (RequestResponse(peer, req, resp)) {
    if (! resp.Json().empty()) {
      cout << resp.Json().dump(2) << '\n';
      rc = true;
    }
  }
  return rc;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static bool Get(Credence::Peer & peer, const string & table,
                const string & addrstr, nlohmann::json & jv)
{
  bool  rc = false;
  Ipv4Address  addr(addrstr);
  McBlock::ResponseMessage  resp;
  McBlock::RequestMessage   req((McBlock::GetReq(table, addr)));
  if (RequestResponse(peer, req, resp)) {
    if (! resp.Json().empty()) {
      jv = resp.Json();
      rc = true;
    }
  }
  return rc;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static bool Edit(Credence::Peer & peer, const string & table,
                 const string & addrstr)
{
  bool            rc = false;
  nlohmann::json  jv;
  if (Get(peer, table, addrstr, jv)) {
    cout << "Editing prefix " << jv["prefix"].get<string>() << " in table "
         << jv["table"].get<string>() << endl;
    string  st;
    string  countryCode;
    while (countryCode.empty()) {
      cout << "countryCode [" << jv["countryCode"].get<string>() << "]: "
           << std::flush;
      std::getline(cin, st);
      if (! st.empty()) {
        if (st.length() == 2) {
          countryCode = st;
        }
      }
      else {
        countryCode = jv["countryCode"].get<string>();
      }
    }
    jv["countryCode"] = countryCode;
  readDaysRemaining:
    cout << "daysRemaining [" << jv["daysRemaining"].get<uint32_t>() << "]: "
         << std::flush;
    std::getline(cin, st);
    if (! st.empty()) {
      try {
        jv["daysRemaining"] = static_cast<uint32_t>(stoul(st, nullptr, 10));
      }
      catch (...) {
        goto readDaysRemaining;
      }
    }

    jv["type"] = "edit";
    McBlock::ResponseMessage  resp;
    McBlock::RequestMessage   req(jv);
    if (RequestResponse(peer, req, resp)) {
      if (! resp.Json().empty()) {
        cout << resp.Json().dump(2) << '\n';
        rc = true;
      }
    }
  }
  return rc;
}


//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static bool Get(Credence::Peer & peer, const string & table,
                const string & addrstr)
{
  bool  rc = false;
  Ipv4Address  addr(addrstr);
  McBlock::ResponseMessage  resp;
  McBlock::RequestMessage   req((McBlock::GetReq(table, addr)));
  if (RequestResponse(peer, req, resp)) {
    if (! resp.Json().empty()) {
      cout << resp.Json().dump(2) << '\n';
      rc = true;
    }
  }
  return rc;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
int main(int argc, char *argv[])
{
  int      rc = 1;
  string   mcblockdHost;
  OptArgs  optargs;
  optargs.AddOptArg("h:", "host", false, "", "server host");
  
  int  nextArg = optargs.Parse(argc, argv);

  if (nextArg > (argc - 1)) {
    Usage(argv[0]);
    exit(1);
  }
  
  SysLogger::Open("mcblockc", LOG_PERROR, LOG_USER);
  SysLogger::MinimumPriority("info");

  mcblockdHost = optargs.Get<string>('h');
  if (mcblockdHost.empty()) {
    if (getenv("MCBLOCKDHOST") != nullptr) {
      mcblockdHost = getenv("MCBLOCKDHOST");
    }
    else {
      cerr << "mcblockd host required (or MCBLOCKDHOST set in environment)\n";
      Usage(argv[0]);
      exit(1);
    }
  }

  Signal  sigPipe(SIGPIPE);
  sigPipe.Block();

  Credence::Peer       peer;
  Credence::KeyStash   keyStash;
  Credence::KnownKeys  knownKeys;
  if (Connect(mcblockdHost, peer)) {
    if (! peer.Authenticate(keyStash, knownKeys)) {
      cerr << "Failed to authenticate!\n";
      return 1;
    }
  }
  else {
    cerr << "Failed to connect to " << mcblockdHost << '\n';
    return 1;
  }

  if (string(argv[nextArg]) == "ping") {
    if (Ping(peer)) {
      rc = 0;
    }
  }
  else if (string(argv[nextArg]) == "activate") {
    if (argc > ++nextArg) {
      string  table = argv[nextArg];
      if (argc > ++nextArg) {
        vector<Ipv4Prefix>  prefixes;
        for ( ; nextArg < argc; ++nextArg) {
          prefixes.push_back(Ipv4Prefix(argv[nextArg]));
        }
        if (ActivatePrefixes(peer, table, prefixes)) {
          rc = 0;
        }
      }
      else {
        Usage(argv[0]);
        exit(1);
      }
    }
    else {
      Usage(argv[0]);
      exit(1);
    }
  }
  else if (string(argv[nextArg]) == "deactivate") {
    if (argc > ++nextArg) {
      string  table = argv[nextArg];
      if (argc > ++nextArg) {
        vector<Ipv4Prefix>  prefixes;
        for ( ; nextArg < argc; ++nextArg) {
          prefixes.push_back(Ipv4Prefix(argv[nextArg]));
        }
        if (DeactivatePrefixes(peer, table, prefixes)) {
          rc = 0;
        }
      }
      else {
        Usage(argv[0]);
        exit(1);
      }
    }
    else {
      Usage(argv[0]);
      exit(1);
    }
  }
  else if (string(argv[nextArg]) == "search") {
    if (argc > ++nextArg) {
      if (Search(peer, Ipv4Address(argv[nextArg]))) {
        rc = 0;
      }
    }
    else {
      Usage(argv[0]);
      exit(1);
    }
  }
  else if (string(argv[nextArg]) == "getactive") {
    string  table;
    if (argc > ++nextArg) {
      table = argv[nextArg];
    }
    if (GetActivePrefixes(peer, table)) {
      rc = 0;
    }
    else {
      Usage(argv[0]);
      exit(1);
    }
  }
  else if (string(argv[nextArg]) == "getaddrules") {
    if (argc > ++nextArg) {
      string  table = argv[nextArg];
      if (GetAddRules(peer, table)) {
        rc = 0;
      }
    }
    else {
      Usage(argv[0]);
      exit(1);
    }
  }
  else if (string(argv[nextArg]) == "loghit") {
    if (argc > ++nextArg) {
      string  table = argv[nextArg];
      if (argc > ++nextArg) {
        string  hostaddr = argv[nextArg];
        if (LogHit(peer, table, hostaddr)) {
          rc = 0;
        }
      }
      else {
        Usage(argv[0]);
        exit(1);
      }
    }
    else {
      Usage(argv[0]);
      exit(1);
    }
  }
  else if (string(argv[nextArg]) == "get") {
    if (argc > ++nextArg) {
      string  table = argv[nextArg];
      if (argc > ++nextArg) {
        string  hostaddr = argv[nextArg];
        if (Get(peer, table, hostaddr)) {
          rc = 0;
        }
      }
      else {
        Usage(argv[0]);
        exit(1);
      }
    }
    else {
      Usage(argv[0]);
      exit(1);
    }
  }
  else if (string(argv[nextArg]) == "edit") {
    if (argc > ++nextArg) {
      string  table = argv[nextArg];
      if (argc > ++nextArg) {
        string  hostaddr = argv[nextArg];
        if (Edit(peer, table, hostaddr)) {
          rc = 0;
        }
      }
      else {
        Usage(argv[0]);
        exit(1);
      }
    }
    else {
      Usage(argv[0]);
      exit(1);
    }
  }
  
  else {
    Usage(argv[0]);
    exit(1);
  }

  return rc;
}
