//===========================================================================
// @(#) $Name$
// @(#) $Id: DwmMcBlockDb.cc 9376 2017-05-23 04:37:38Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2015
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmMcBlockDb.cc
//!  \brief NOT YET DOCUMENTED
//---------------------------------------------------------------------------

extern "C" {
  #include <fcntl.h>
}

#include "DwmCvsTag.hh"
#include "DwmSysLogger.hh"

#include "DwmMcBlockDb.hh"

static const Dwm::CvsTag cvstag("@(#) $Name$ $Id: DwmMcBlockDb.cc 9376 2017-05-23 04:37:38Z dwm $");

using namespace std;

namespace Dwm {

  namespace McBlock {

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Db::FindContained(const Ipv4Prefix & prefix,
                           vector<pair<Ipv4Prefix,DbEntry> > & contained) const
    {
      contained.clear();
      for (uint8_t  maskLen = prefix.MaskLength();
           maskLen < 33; ++maskLen) {
        auto hm = _hashMaps[maskLen];
        auto  it = hm.lower_bound(prefix.Network());
        if (it != hm.end()) {
          while (prefix.Contains(it->first) && (it != hm.end())) {
            pair<Ipv4Prefix,DbEntry>  entry = {
              Ipv4Prefix(it->first, maskLen),
              it->second
            };
            contained.push_back(entry);
            ++it;
          }
        }
      }
      return (! contained.empty());
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Db::NextWidest(const Ipv4Prefix & prefix,
                        typename _RepSubType::iterator & nextWidest)
    {
      bool  rc = false;
      for (int8_t maskLen = prefix.MaskLength() - 1;
           maskLen >= 0; --maskLen) {
        if (! _hashMaps[maskLen].empty()) {
          Ipv4Prefix  widerPrefix(prefix.Network(), maskLen);
          auto it = _hashMaps[maskLen].find(widerPrefix.Network());
          if (it != _hashMaps[maskLen].end()) {
            rc = true;
            nextWidest = it;
            break;
          }
        }
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Db::Coalesce()
    {
      bool  rc = false;
      if (CombineAdjacents()) {
        rc = true;
      }
      if (CombineCovered()) {
        rc = true;
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Db::CombineCovered()
    {
      bool  rc = false;
      for (uint8_t maskLen = 32; maskLen > 0; --maskLen) {
        auto & hm = _hashMaps[maskLen];
        if (! hm.empty()) {
          vector<Ipv4Address>  deleteNets;
          for (auto it : hm) {
            typename _RepSubType::iterator  widerEntry;
            if (NextWidest(Ipv4Prefix(it.first, maskLen), widerEntry)) {
              deleteNets.push_back(it.first);
              if (it.second.IsActive()) {
                if (! widerEntry->second.IsActive()) {
                  widerEntry->second.Interval(it.second.Interval());
                }
              }
              rc = true;
            }
          }
          if (! deleteNets.empty()) {
            for (auto dn : deleteNets) {
              hm.erase(dn);
            }
          }
        }
      }
      return rc;
    }
    
    //--------------------------------------------------------------------
    //!  
    //--------------------------------------------------------------------
    bool Db::CombineAdjacents()
    {
      bool  rc = false;
      for (uint8_t maskLen = 32; maskLen > 0; --maskLen) {
        auto & hm = _hashMaps[maskLen];
        if (! hm.empty()) {
          vector<Ipv4Address>  deleteNets;
          auto  it = hm.begin();
          for ( ; it != hm.end(); ++it) {
            if (! ((ntohl(it->first.Raw()) >> (32 - maskLen)) & 0x01)) {
              auto  nextit = it;
              ++nextit;
              if (nextit != hm.end()) {
                Ipv4Prefix  widerPrefix(it->first, maskLen - 1);
                if (widerPrefix.Contains(nextit->first)) {
                  //  does wider entry exist?
                  auto  widerit = _hashMaps[maskLen-1].find(widerPrefix.Network());
                  if (widerit != _hashMaps[maskLen-1].end()) {
                    //  Wider entry exists.  is it active?
                    if (! widerit->second.IsActive()) {
                      //  Wider entry inactive.  If either narrower entry is
                      //  active, activate the wider entry.
                      if (it->second.IsActive() || nextit->second.IsActive()) {
                        TimeValue  startTime(true);
                        TimeValue  endTime(startTime);
                        if (it->second.IsActive()
                            && (it->second.Interval().End() > endTime)) {
                          endTime = it->second.Interval().End();
                        }
                        if (nextit->second.IsActive()
                            && (nextit->second.Interval().End() > endTime)) {
                          endTime = nextit->second.Interval().End();
                        }
                        widerit->second.Interval(TimeInterval(startTime, endTime));
                      }
                      //  Mark narrower entries for deletion.
                      deleteNets.push_back(it->first);
                      deleteNets.push_back(nextit->first);
                      rc = true;
                    }
                  }
                  else {
                    //  Wider entry does not exist.
                    DbEntry    newEntry(widerPrefix);
                    TimeValue  startTime(true);
                    TimeValue  endTime(startTime);
                    endTime.Set(endTime.Secs() + (30 * 24 * 60 * 60), 0);
                    if (it->second.Interval().End() > endTime) {
                      endTime = it->second.Interval().End();
                    }
                    if (nextit->second.Interval().End() > endTime) {
                      endTime = nextit->second.Interval().End();
                    }
                    //  Insert wider entry.
                    newEntry.Interval(TimeInterval(startTime, endTime));
                    _hashMaps[maskLen-1][widerPrefix.Network()] = newEntry;
                    //  Mark narrower entries for deletion.
                    deleteNets.push_back(it->first);
                    deleteNets.push_back(nextit->first);
                    rc = true;
                  }
                }
                else {
                  //  Is there a wider prefix?
                  
                }
                
              }
            }
          }
          if (! deleteNets.empty()) {
            for (auto dn : deleteNets) {
              _hashMaps[maskLen].erase(dn);
            }
          }
        }
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Db::Load(const string & filename)
    {
      _dbFilename = filename;
      bool  rc = false;
      int  fd = open(filename.c_str(), O_RDONLY|O_SHLOCK);
      if (fd >= 0) {
        if (Read(fd) >= 0) {
          rc = true;
        }
        close(fd);
      }
      else {
        Syslog(LOG_ERR, "Failed to open+lock '%s': %m", filename.c_str());
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Db::Save(const string & filename) const
    {
      bool  rc = false;
      string  dbfn(filename);
      if (dbfn.empty()) {
        dbfn = _dbFilename;
      }
      if (! dbfn.empty()) {
        int  fd = open(dbfn.c_str(),
                       O_CREAT|O_WRONLY|O_TRUNC|O_EXLOCK, 0644);
        if (fd >= 0) {
          Write(fd);
          close(fd);
          rc = true;
          Syslog(LOG_INFO, "Saved database to %s", dbfn.c_str());
        }
        else {
          Syslog(LOG_ERR, "Failed to open+lock '%s': %m", dbfn.c_str());
        }
      }
      else {
        Syslog(LOG_ERR, "Empty filename in Db::Save()!");
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Db::GetByActive(std::vector<DbEntry> & target, bool active) const
    {
      target.clear();
      vector<pair<Ipv4Prefix,DbEntry>> allEntries;
      SortByKey(allEntries, true);
      for (auto i : allEntries) {
        if (i.second.IsActive() == active) {
          target.push_back(i.second);
        }
      }
      return (! target.empty());
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Db::GetAllActive(std::vector<DbEntry> & target) const
    {
      return GetByActive(target, true);
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Db::GetAllInactive(std::vector<DbEntry> & target) const
    {
      return GetByActive(target, false);
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void Db::DeleteInactive()
    {
      lock_guard<shared_mutex>  lock(_mtx);
      for (uint8_t maskLen = 32; maskLen > 0; --maskLen) {
        auto & hm = _hashMaps[maskLen];
        if (! hm.empty()) {
          auto  it = hm.begin();
          while (it != hm.end()) {
            if (! it->second.IsActive()) {
              Syslog(LOG_INFO, "Deleting inactive db entry %s (%s)",
                     it->second.Prefix().ToShortString().c_str(),
                     it->second.Country().c_str());
              it = hm.erase(it);
            }
            else {
              ++it;
            }
          }
        }
      }
      return;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void Db::GetMinimalPfList(vector<Ipv4Prefix> & prefixes) const
    {
      if (! prefixes.empty()) {
        prefixes.clear();
      }
      std::shared_lock<std::shared_mutex>  lock(_mtx);
      for (uint8_t maskLen = 32; maskLen > 1; --maskLen) {
        auto & hm = _hashMaps[maskLen];
        if (! hm.empty()) {
          auto  it = hm.begin();
          for ( ; it != hm.end(); ++it) {
            bool  foundWider = false;
            for (uint8_t i = maskLen - 1; i > 0; --i) {
              auto  & widerMap = _hashMaps[i];
              Ipv4Prefix  widerPfx(it->first, i);
              if (widerMap.find(widerPfx.Network()) != widerMap.end()) {
                foundWider = true;
                break;
              }
            }
            if ((! foundWider) && it->second.IsActive()) {
              prefixes.push_back(Ipv4Prefix(it->first, maskLen));
            }
          }
        }
      }
      return;
    }
    
    
  }  // namespace McBlock

}  // namespace Dwm
