//===========================================================================
// @(#) $Name$
// @(#) $Id: TestDwmMcBlockDb.cc 9194 2017-04-21 00:04:59Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2015
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file TestDwmMcBlockDb.cc
//!  \brief NOT YET DOCUMENTED
//---------------------------------------------------------------------------

#include <fstream>
#include <sstream>

#include "DwmUnitAssert.hh"
#include "DwmMcBlockDb.hh"

using namespace std;
using namespace Dwm;

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
bool LoadDbPrefixes(McBlock::Db & db)
{
  bool  rc = false;
  ifstream  is("./prefixes.txt");
  if (is) {
    char  buf[512];
    McBlock::DbEntry  dbEntry;
    while (is.getline(buf, 512, '\n')) {
      Ipv4Prefix  pfx(buf);
      dbEntry.Prefix(pfx);
      db.AddEntry(dbEntry);
      memset(buf, 0, 512);
    }
    is.close();
    rc = true;
  }
  return rc;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
void TestIOStream()
{
  string  s;
  ostringstream  os(s);
  McBlock::Db  db;
  if (LoadDbPrefixes(db)) {
    UnitAssert(db.Write(os));
    UnitAssert(os.tellp() == db.StreamedLength());
    {
      istringstream  is(os.str());
      McBlock::Db  db2;
      UnitAssert(db2.Read(is));
      UnitAssert(is.tellg() == db.StreamedLength());
      UnitAssert(db == db2);
    }

    vector<pair<Ipv4Prefix,McBlock::DbEntry>>  vec;
    db.SortByKey(vec);
    McBlock::DbEntry  dbEntry;
    for (auto i : vec) {
      UnitAssert(db.Find(i.first, dbEntry));
    }
  }
  return;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
void TestContained()
{
  McBlock::Db  db;

  McBlock::DbEntry  dbEntry;
  set<Ipv4Prefix>   prefixes = {
    Ipv4Prefix("10.1.1.0/25"),
    Ipv4Prefix("10.1.1.128/25"),
    Ipv4Prefix("10.1.2/24"),
    Ipv4Prefix("10.1.0/17"),
    Ipv4Prefix("10.1.128/17"),
    Ipv4Prefix("10.1/16")
  };

  for (auto pi : prefixes) {
    db.Add(pi, dbEntry);
  }
  
  vector<pair<Ipv4Prefix,McBlock::DbEntry> >  matches;
  UnitAssert(db.FindContained(Ipv4Prefix("10.1/16"), matches));
  UnitAssert(matches.size() == prefixes.size());
  for (auto i : matches) {
    UnitAssert(prefixes.find(i.first) != prefixes.end());
  }
    
  return;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
void TestCoalesce()
{
  McBlock::Db  db;

  McBlock::DbEntry  dbEntry;
  set<Ipv4Prefix>   prefixes = {
    Ipv4Prefix("10.1.1.0/25"),
    Ipv4Prefix("10.1.1.128/25"),
    Ipv4Prefix("10.1.2/24"),
    Ipv4Prefix("10.1.3/24"),
    Ipv4Prefix("10.1.0/17"),
    Ipv4Prefix("10.1.128/17"),
    Ipv4Prefix("10.1/16"),
    Ipv4Prefix("10.2.128/17")
  };

  for (auto pi : prefixes) {
    db.Add(pi, dbEntry);
  }

  UnitAssert(db.Size() == 8);
  db.Coalesce();
  UnitAssert(db.Size() == 2);
  return;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
void TestMinimalPf()
{
  McBlock::Db  db;

  McBlock::DbEntry  dbEntry;
  TimeValue         now(true);
  TimeValue         later(now.Secs() + 24 * 60 * 60, 0);
  dbEntry.Interval(TimeInterval(now, later));
  
  set<Ipv4Prefix>   prefixes = {
    Ipv4Prefix("10.1.1.0/25"),
    Ipv4Prefix("10.1.1.128/25"),
    Ipv4Prefix("10.1.2/24"),
    Ipv4Prefix("10.1.3/24"),
    Ipv4Prefix("10.1.0/17"),
    Ipv4Prefix("10.1.128/17"),
    Ipv4Prefix("10.1/16"),
    Ipv4Prefix("10.2.128/17")
  };

  for (auto pi : prefixes) {
    db.Add(pi, dbEntry);
  }
  vector<Ipv4Prefix>  minimalList;
  db.GetMinimalPfList(minimalList);
  UnitAssert(minimalList.size() == 2);
  UnitAssert(find_if(minimalList.begin(), minimalList.end(),
                     [&] (const Ipv4Prefix & pfx)
                     { return (pfx == Ipv4Prefix("10.1/16")); })
             != minimalList.end());
  UnitAssert(find_if(minimalList.begin(), minimalList.end(),
                     [&] (const Ipv4Prefix & pfx)
                     { return (pfx == Ipv4Prefix("10.2.128/17")); })
             != minimalList.end());
  return;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
int main(int argc, char *argv[])
{
  TestIOStream();
  TestContained();
  TestCoalesce();
  TestMinimalPf();
  
  if (Dwm::Assertions::Total().Failed() > 0) {
    Dwm::Assertions::Print(std::cerr, true);
    return 1;
  }
  else {
    std::cout << Dwm::Assertions::Total() << " passed" << std::endl;
  }

  return 0;
}
