//===========================================================================
// @(#) $Name:$
// @(#) $Id: DwmMcBlockResponder.hh 12351 2024-05-03 09:12:30Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2017
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmMcBlockResponder.hh
//!  \brief Dwm::McBlock::Responder class definition
//---------------------------------------------------------------------------

#ifndef _DWMMCBLOCKRESPONDER_HH_
#define _DWMMCBLOCKRESPONDER_HH_

#include <atomic>
#include <string>
#include <thread>
#include <nlohmann/json.hpp>

#include "DwmCredencePeer.hh"
#include "DwmRDAPFetcher.hh"
#include "DwmMcBlockDb.hh"
#include "DwmMcBlockServerConfig.hh"
#include "DwmMcBlockAddRules.hh"
#include "DwmMcBlockLogEntryTrackers.hh"
#include "DwmPfDevice.hh"

namespace Dwm {

  namespace McBlock {

    class Server;
    
    //------------------------------------------------------------------------
    //!  Encapsulates a thread that responds to requests from a single
    //!  client.
    //------------------------------------------------------------------------
    class Responder
    {
    public:
      Responder(boost::asio::ip::tcp::socket && s, Server & server);
      bool Stop();
      bool Join();
      
    private:
      Credence::Peer      _peer;
      sockaddr_in         _clientAddr;
      std::string         _theirId;
      Server             &_server;
      std::string         _agreedKey;
      std::atomic<bool>   _run;
      std::thread         _thread;
      std::atomic<bool>   _running;

      static bool IsValidAddRequest(const nlohmann::json & request);
      static bool IsValidGetRequest(const nlohmann::json & request);
      static bool IsValidEditRequest(const nlohmann::json & request);
      static bool IsValidActivateRequest(const nlohmann::json & request);
      static bool IsValidDeactivateRequest(const nlohmann::json & request);
      static bool IsValidDeleteRequest(const nlohmann::json & request);
      static bool IsValidGetActiveRequest(const nlohmann::json & request);
      static bool IsValidSearchRequest(const nlohmann::json & request);
      static bool IsValidGetAddRulesRequest(const nlohmann::json & request);
      static bool IsValidLogHitRequest(const nlohmann::json & request);
      static bool IsValidPingRequest(const nlohmann::json & request);
      void GetRDAPEntry(const Ipv4Address & addr, Ipv4Prefix & prefix,
                        std::string & country);
      nlohmann::json HandleAddRequest(const nlohmann::json & request);
      nlohmann::json HandleDeleteRequest(const nlohmann::json & request);
      nlohmann::json HandleGetRequest(const nlohmann::json & request);
      nlohmann::json HandleEditRequest(const nlohmann::json & request);
      nlohmann::json HandleActivateRequest(const nlohmann::json & request);
      nlohmann::json HandleDeactivateRequest(const nlohmann::json & request);
      nlohmann::json HandleGetActiveRequest(const nlohmann::json & request);
      nlohmann::json HandleSearchRequest(const nlohmann::json & request);
      nlohmann::json HandleGetAddRulesRequest(const nlohmann::json & request);
      nlohmann::json HandleLogHitRequest(const nlohmann::json & request);
      nlohmann::json HandlePingRequest(const nlohmann::json & request);
      nlohmann::json HandleRequest(const nlohmann::json & request);
      void AddToDatabase(const std::string & tableName,
                         const Ipv4Prefix & prefix, uint32_t days,
                         const std::string & country);
      void AddToDatabase(const std::string & tableName,
                         const Ipv4Prefix & prefix);
      bool ModifyDatabaseEntry(const std::string & tableName,
                               const Ipv4Prefix & prefix,
                               uint32_t days, const std::string & country);
      bool RemoveFromDatabase(const std::string & tableName,
                              const Ipv4Prefix & prefix);
      void SaveDatabase(const std::string & tableName);
      bool KillSourceState(const Ipv4Prefix & prefix);
      
      RDAP::FetchedEntryMap
      GetRDAPEntries(const std::vector<Ipv4Address> & addrs);
      
      void Run();
    };
      
  }  // namespace McBlock

}  // namespace Dwm

#endif  // _DWMMCBLOCKRESPONDER_HH_
