//===========================================================================
// @(#) $Name:$
// @(#) $Id: DwmMcBlockServer.cc 12269 2023-10-27 02:50:30Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2017
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmMcBlockServer.cc
//!  \brief Dwm::McBlock::Server class implementation
//---------------------------------------------------------------------------

extern "C" {
  #include <sys/types.h>
  #include <sys/socket.h>
  #include <netinet/in.h>
  #include <arpa/inet.h>
#ifndef __APPLE__
  #include <pthread_np.h>
#endif
}

#include <fstream>
#include <sstream>

#include <boost/asio.hpp>

#include "DwmSvnTag.hh"
#include "DwmSysLogger.hh"
#include "DwmTimeValue64.hh"
#include "DwmMcBlockPortability.hh"
#include "DwmMcBlockServer.hh"
#undef v4

#include "DwmMcBlockResponseMessage.hh"

static const Dwm::SvnTag  svntag("@(#) $DwmPath: dwm/mcplex/mcblock/tags/mcblock-0.3.6/apps/mcblockd/DwmMcBlockServer.cc 12269 $");

using namespace std;

namespace Dwm {

  namespace McBlock {

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    Server::Server(const Cfg2Json & config)
        : _config(config), _addRules(config), _pfdev(_config.Device()),
          _keyStash(_config.KeyDirectory()),
          _knownKeys(_config.KeyDirectory()),
          _run(false), _acceptors(), _thread(), _responders(),
          _logEntryTrackers(), _databases()
    {
      for (auto & rft : _addRules.RulesForTables()) {
        string  dbFileName(_config.DatabaseDirectory() + "/pf." + rft.first
                           + ".db");
        _databases[rft.first].Load(dbFileName);
      }
      _nextListSaveTime = chrono::system_clock::now() + chrono::hours(24);

      namespace ip = boost::asio::ip;
      using boost::asio::ip::tcp;
      boost::system::error_code  ec;
      ip::address  addr =
        ip::address::from_string((std::string)_config.TCPAddr(), ec);
      Syslog(LOG_INFO, "Server address %s",
             ((std::string)(_config.TCPAddr())).c_str());
      if (! ec) {
        tcp::endpoint  ep(addr, _config.TCPPort());
        ip::tcp::acceptor  acc(_ioContext.get_executor());
        acc.open(tcp::v4());
        boost::asio::socket_base::reuse_address option(true);
        acc.set_option(option);
        acc.bind(ep);
        auto & a = _acceptors.emplace_back(std::move(acc));
        a.listen(20);
        AcceptLoop(a);
      }
      else {
        Syslog(LOG_ERR, "Bad TCP Address %s",
               ((std::string)(_config.TCPAddr())).c_str());
        exit(1);
      }
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const ServerConfig & Server::Config() const
    {
      return _config;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const AddRules & Server::GetAddRules() const
    {
      return _addRules;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const Pf::Device & Server::GetPfDevice() const
    {
      return _pfdev;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    LogEntryTrackers & Server::GetLogEntryTrackers()
    {
      return _logEntryTrackers;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    std::map<std::string,Db> & Server::GetDatabases()
    {
      return _databases;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Server::Start()
    {
      _run = true;
      _thread = std::thread(&Server::Run, this);
      return true;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void Server::StopResponders()
    {
      for (auto ri = _responders.begin(); ri != _responders.end(); ) {
        if ((*ri)->Stop()) {
          ri = _responders.erase(ri++);
        }
        else {
          ++ri;
        }
      }
      return;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void Server::Stop()
    {
      Syslog(LOG_INFO, "Stopping server");
      _run = false;
      // StopResponders();
      for (auto & a : _acceptors) {
        a.cancel();
      }
      Syslog(LOG_INFO, "Server stopped");
      if (_thread.joinable()) {
        _thread.join();
      }
      SavePfLists();
      return;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void Server::CleanupResponders()
    {
      for (auto ri = _responders.begin(); ri != _responders.end(); ) {
        if ((*ri)->Join()) {
          ri = _responders.erase(ri);
        }
        else {
          ++ri;
        }
      }
      return;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void Server::CleanupLogTrackers()
    {
      for (auto & tr : _logEntryTrackers) {
        auto ari = _addRules.RulesForTable(tr.first);
        if (ari != _addRules.RulesForTables().end()) {
          tr.second.ClearOld(ari->second);
        }
      }
      return;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void Server::Run()
    {
      Syslog(LOG_INFO, "Server thread started");
      set_pthread_name(_thread.native_handle(), "server");
      _pfdev.Open();
      CleanupInactiveDbEntries();
      SavePfLists();

      while (_run) {
        _ioContext.run();
        this_thread::sleep_for(chrono::milliseconds(100));
      }
      Syslog(LOG_INFO, "Server thread done");
      return;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void Server::CleanupInactiveDbEntries()
    {
      for (auto & db : _databases) {
        //  Remove pf table entries for inactive db entries
        vector<DbEntry>  inactiveEntries;
        db.second.GetAllInactive(inactiveEntries);
        Pf::Table  table(_pfdev, "", db.first);
        for (auto & entry : inactiveEntries) {
          table.Remove(entry.Prefix());
        }
        //  Remove inactive entries from database
        db.second.DeleteInactive();
        //  Save the database
        db.second.Save("");
      }
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void Server::SavePfLists() const
    {
      for (auto & db : _databases) {
        string  fileName(_config.DatabaseDirectory() + "/pf." + db.first);
        ofstream  os(fileName.c_str());
        if (os) {
          vector<Ipv4Prefix>  pfxList;
          db.second.GetMinimalPfList(pfxList);
          if (! pfxList.empty()) {
            for (auto pfx : pfxList) {
              os << pfx.ToShortString() << '\n';
            }
          }
          os.close();
        }
      }
      return;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void Server::AcceptLoop(boost::asio::ip::tcp::acceptor & a)
    {
      using boost::asio::ip::tcp;
      
      a.async_accept(a.get_executor(),
                     [this,&a](error_code ec, tcp::socket && s)
                     {
                       CleanupResponders();
                       if (! ec) {
                         _responders.push_back(std::make_shared<Responder>(std::move(s), *this));
                         AcceptLoop(a);
                       }
                       else {
                         Syslog(LOG_ERR, "Exiting AcceptLoop(): %s",
                                ec.message().c_str());
                       }
                     });
    }

#if 0
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void Server::SendMcastStatus()
    {
      static TimeValue64  lastSend(true);
      TimeValue64         now(true);
      TimeValue64         elapsed(now);
      elapsed -= lastSend;
      if (elapsed > TimeValue64(0, 500000)) {
        Json::FastWriter  fw;
        Auth::SymCrypto::Message  msg(_mySecret,
                                      fw.write(_doors.StatusJson()));
        if (msg.SendTo(_mcastSocket, 0, _config.MulticastAddr(),
                       _config.MulticastPort())) {
          lastSend = now;
        }
      }
      return;
    }
#endif

  }  // namespace McBlock

}  // namespace Dwm
