//===========================================================================
// @(#) $Name$
// @(#) $Id: DwmMcBlockDb.hh 12351 2024-05-03 09:12:30Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2015
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmMcBlockDb.hh
//!  \brief NOT YET DOCUMENTED
//---------------------------------------------------------------------------

#ifndef _DWMMCBLOCKDB_HH_
#define _DWMMCBLOCKDB_HH_

#include "DwmIO.hh"
#include "DwmGZIO.hh"
#include "DwmMcBlockIpv4Routes.hh"
#include "DwmMcBlockDbEntry.hh"

namespace Dwm {

  namespace McBlock {

    //------------------------------------------------------------------------
    //!  Encapsulates a database of DbEntry values keyed by Ipv4Prefix.
    //------------------------------------------------------------------------
    class Db
      : public Ipv4Routes<DbEntry>
    {
    public:
      //----------------------------------------------------------------------
      //!  Constructor
      //----------------------------------------------------------------------
      Db()
          : Ipv4Routes<DbEntry>(), _dbFilename()
      {}

      //----------------------------------------------------------------------
      //!  Adds an entry to the database.  Returns true on success, false
      //!  on failure.  A reason for failure: the entry's prefix is already
      //!  in the database.
      //----------------------------------------------------------------------
      bool AddEntry(const DbEntry & entry)
      {
        return Add(entry.Prefix(), entry);
      }

      //----------------------------------------------------------------------
      //!  Deletes an entry from the databse.  Returns true on success,
      //!  false on failure.  A reason for failure: there is no entry in
      //!  the database for the given prefix.
      //----------------------------------------------------------------------
      bool DeleteEntry(const Ipv4Prefix & prefix)
      {
        return Delete(prefix);
      }

      // bool Coalesce();
      
      //----------------------------------------------------------------------
      //!  Finds all entries whose prefixes are covered by the given @c
      //!  prefix.  Places the results in @c contained and returns true if
      //!  any contained entries were found, else returns false.
      //----------------------------------------------------------------------
      bool
      FindContained(const Ipv4Prefix & prefix,
                    std::vector<std::pair<Ipv4Prefix,DbEntry> > & contained) const;
        
      //----------------------------------------------------------------------
      //!  Loads the database from the given @c filename.  Returns true
      //!  on success, false on failure.
      //----------------------------------------------------------------------
      bool Load(const std::string & filename);

      //----------------------------------------------------------------------
      //!  Saves the database to the given @c filename.  Returns true on
      //!  success, false on failure.
      //----------------------------------------------------------------------
      bool Save(const std::string & filename) const;

      //----------------------------------------------------------------------
      //!  Populates @c target with all of the currently active entries
      //!  from the database.  Returns true if any active entries were
      //!  found.
      //----------------------------------------------------------------------
      bool GetAllActive(std::vector<DbEntry> & target) const;
      
      //----------------------------------------------------------------------
      //!  Populates @c target with all of the currently inactive entries
      //!  from the database.  Returns true if any inactive entries were
      //!  found.
      //----------------------------------------------------------------------
      bool GetAllInactive(std::vector<DbEntry> & target) const;

      //----------------------------------------------------------------------
      //!  Delete all inactive entries.
      //----------------------------------------------------------------------
      void DeleteInactive();

      //----------------------------------------------------------------------
      //!  Populates @c prefixes with a minimal pf list.
      //----------------------------------------------------------------------
      void GetMinimalPfList(std::vector<Ipv4Prefix> & prefixes) const;
      
      //----------------------------------------------------------------------
      //!  Prints a database's entries in human-readable to an ostream.
      //----------------------------------------------------------------------
      friend std::ostream & operator << (std::ostream & os, const Db & db);
                                         
    private:
      std::string  _dbFilename;
      
      bool GetByActive(std::vector<DbEntry> & target, bool active) const;
      bool NextWidest(const Ipv4Prefix & prefix,
                      typename _RepSubType::iterator & nextWidest);
      bool CombineCovered();
      bool CombineAdjacents();
    };
    
  }  // namespace McBlock

}  // namespace Dwm

#endif  // _DWMMCBLOCKDB_HH_
