%{
  #include <cstdio>
  #include <map>
  #include <regex>

  #include "DwmIpv4Address.hh"
  #include "DwmIpv6Address.hh"
  #include "DwmSvnTag.hh"
  #include "DwmSysLogger.hh"
  #include "DwmMcroverConfig.hh"
  #include "DwmMcroverConfigParse.hh"

  extern std::string  g_configPath;
  
  extern "C" {
    #include <stdarg.h>

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void mcrovercfgerror(const char *arg, ...)
    {
      va_list  ap;
      va_start(ap, arg);
      char  *s = nullptr;
      if (vasprintf(&s, arg, ap) != -1) {
        Syslog(LOG_ERR, "%s: '%s' at line %d of %s\n",
               s, yytext, yylineno, g_configPath.c_str());
      }
      return;
    }
  }

  static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/mcplex/mcrover/tags/mcrover-0.1.12/classes/src/DwmMcroverConfigLex.lex 12283 $");

  //--------------------------------------------------------------------------
  //!  
  //--------------------------------------------------------------------------
  static const std::map<std::string,int>  g_configKeywords = {
    { "BANDITS",             BANDITS        },
    { "CREDENCE",            CREDENCE       },
    { "DISKS",               DISKS          },
    { "DNS",                 DNS            },
    { "DUMPS",               DUMPS          },
    { "FS",                  FS             },
    { "GUESTS",              GUESTS         },
    { "RPC",                 RPC            },
    { "RPC6",                RPC6           },
    { "ROUTES",              ROUTES         },
    { "SMTP",                SMTP           },
    { "SMTP6",               SMTP6          },
    { "UPS",                 UPS            },
    { "WEB",                 WEB            },
    { "WEBAPP",              WEBAPP         },
    { "ZFS",                 ZFS            },
    { "ZPOOL",               ZPOOL          },
    { "addresses",           ADDRESSES      },
    { "address",             ADDRESS        },
    { "alerts",              ALERTS         },
    { "allowedClients",      ALLOWEDCLIENTS },
    { "appname",             APPNAME        },
    { "capacity",            CAPACITY       },
    { "device",              DEVICE         },
    { "facility",            FACILITY       },
    { "false",               FALSESTRING    },
    { "host",                HOST           },
    { "ipv4",                IPV4           },
    { "ipv6",                IPV6           },
    { "keyDirectory",        KEYDIRECTORY   },
    { "latitude",            LATITUDE       },
    { "level",               LEVEL          },
    { "local",               LOCAL          },
    { "location",            LOCATION       },
    { "logLocations",        LOGLOCATIONS   },
    { "longitude",           LONGITUDE      },
    { "mount",               MOUNT          },
    { "name",                NAME           },
    { "members",             MEMBERS        },
    { "otherpacks",          OTHERPACKS     },
    { "pack",                PACK           },
    { "params",              PARAMS         },
    { "period",              PERIOD         },
    { "port",                PORT           },
    { "protocol",            PROTOCOL       },
    { "qname",               QNAME          },
    { "qtype",               QTYPE          },
    { "resolve",             RESOLVE        },
    { "servers",             SERVERS        },
    { "service",             SERVICE        },
    { "stations",            STATIONS       },
    { "store",               STORE          },
    { "status",              STATUS         },
    { "syslog",              SYSLOG         },
    { "tcp4",                TCP4           },
    { "tcp6",                TCP6           },
    { "tcp4Denied",          TCP4DENIED     },
    { "tcp6Denied",          TCP6DENIED     },
    { "true",                TRUESTRING     },
    { "uri",                 URI            },
    { "validateCertificate", VALIDATECERT   },
    { "version",             VERSION        },
    { "weather",             WEATHER        },
    { "xpath",               XPATH          }
  };

  //--------------------------------------------------------------------------
  //!  
  //--------------------------------------------------------------------------
  static bool IsKeyword(const std::string & s, int & token)
  {
    bool  rc = false;
    auto  it = g_configKeywords.find(s);
    if (g_configKeywords.end() != it) {
      token = it->second;
      rc = true;
    }
    return rc;
  }

  //--------------------------------------------------------------------------
  //!  
  //--------------------------------------------------------------------------
  static bool IsTimePeriod(const std::string & s, uint64_t & dur)
  {
    static const std::vector<std::pair<const char *,uint64_t>>  mults = {
      { "%lluw", 7 * 24 * 60 * 60 },
      { "%llud", 24 * 60 * 60     },
      { "%lluh", 60 * 60          },
      { "%llum", 60               },
      { "%llus", 1                }
    };
  
    bool  rc = false;
    static const std::regex
      rgx("([0-9]+w)?([0-9]+d)?([0-9]+h)?([0-9]+m)?([0-9]+s)?",
          std::regex::ECMAScript|std::regex::optimize);
    std::smatch  sm;
    dur = 0;
    if (regex_match(s, sm, rgx)) {
      if ((mults.size() + 1) == sm.size()) {
        unsigned long long  t;
        for (size_t i = 0; i < mults.size(); ++i) {
          std::string  piece = sm[i+1].str();
          if (! piece.empty()) {
            if (1 == sscanf(piece.c_str(), mults.at(i).first, &t)) {
              dur += t * mults.at(i).second;
              rc = true;
            }
          }
        }
      }
    }
    return rc;
  }
    
  //--------------------------------------------------------------------------
  //!  
  //--------------------------------------------------------------------------
  static bool IsNumber(const std::string & s)
  {
    using std::regex, std::smatch;
    static regex  rgx("[0-9]+", regex::ECMAScript|regex::optimize);
    smatch        sm;
    return std::regex_match(s, sm, rgx);
  }

%}

%option noyywrap
%option prefix="mcrovercfg"
%option yylineno

%x x_quoted
        
%%

<INITIAL>#.*\n
<INITIAL>[^ \t\n\[\]{}=,;"]+       { if (IsNumber(yytext)) {
                                       mcrovercfglval.intVal = atoi(yytext);
                                       return INTEGER;
                                     }
                                     else {
                                       int  token;
                                       if (IsKeyword(yytext, token)) {
                                         return token;
                                       }
                                       else if (IsTimePeriod(yytext,
                                                             mcrovercfglval.uint64Val)) {
                                         return TIMEPERIOD;
                                       }
                                       else {
                                         mcrovercfglval.stringVal =
                                           new std::string(yytext);
                                         return STRING;
                                       }
                                     }
                                   }
<INITIAL>["]                       { BEGIN(x_quoted); }
<x_quoted>[^"]+                    { mcrovercfglval.stringVal =
                                       new std::string(yytext);
                                     return STRING; }
<x_quoted>["]                      { BEGIN(INITIAL); }
<INITIAL>[=,;\[\]\{\}]             { return yytext[0]; }
<INITIAL>[ \t\n]

%%
