//===========================================================================
// @(#) $DwmPath: dwm/mcplex/mcrover/tags/mcrover-0.1.12/classes/src/DwmMcroverRoverAlert.cc 11886 $
// @(#) $Id: DwmMcroverRoverAlert.cc 11886 2021-04-07 05:16:39Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2020
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmMcroverRoverAlert.cc
//!  \author Daniel W. McRobb
//!  \brief Dwm::Mcrover::RoverAlert class implementation
//---------------------------------------------------------------------------

#include <map>

#include "DwmDescriptorIO.hh"
#include "DwmFileIO.hh"
#include "DwmIOUtils.hh"
#include "DwmStreamIO.hh"
#include "DwmIpv4Address.hh"
#include "DwmSvnTag.hh"
#include "DwmMcroverRoverAlert.hh"
#include "DwmMcroverUtils.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/mcplex/mcrover/tags/mcrover-0.1.12/classes/src/DwmMcroverRoverAlert.cc 11886 $");

namespace Dwm {

  namespace Mcrover {

    using namespace std;

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    RoverAlert::RoverAlert()
        : _roverAndFailure(Ipv4Address(INADDR_NONE), e_failUnknown)
    {
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    RoverAlert::RoverAlert(const IpAddress & rover,
                           FailureType failureType)
        : _roverAndFailure(rover, failureType)
    {}
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    string RoverAlert::DisplayString() const
    {
      return ((string)_roverAndFailure.first + ':'
              + FailureName() + " ROVER");
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    nlohmann::json RoverAlert::ToJson() const
    {
      nlohmann::json  j;
      j["rover"] = (string)_roverAndFailure.first;
      j["fail"] = (uint16_t)_roverAndFailure.second;
      return j;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool RoverAlert::FromJson(const nlohmann::json & j)
    {
      bool  rc = false;
      _roverAndFailure.first = Ipv4Address(INADDR_NONE);
      _roverAndFailure.second = e_failUnknown;
      if (j.is_object()) {
        auto  local = j.find("rover");
        if ((local != j.end()) && local->is_string()) {
          _roverAndFailure.first = IpAddress(local->get<string>());
          auto failure = j.find("fail");
          if ((failure != j.end()) && failure->is_number()) {
            _roverAndFailure.second = failure->get<uint16_t>();
            rc = true;
          }
        }
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    istream & RoverAlert::Read(istream & is)
    {
      _roverAndFailure.first = Ipv4Address(INADDR_NONE);
      _roverAndFailure.second = e_failUnknown;
      return StreamIO::Read(is, _roverAndFailure);
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    ssize_t RoverAlert::Read(int fd)
    {
      _roverAndFailure.first = Ipv4Address(INADDR_NONE);
      _roverAndFailure.second = e_failUnknown;
      return DescriptorIO::Read(fd, _roverAndFailure);
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    size_t RoverAlert::Read(FILE * f)
    {
      _roverAndFailure.first = Ipv4Address(INADDR_NONE);
      _roverAndFailure.second = e_failUnknown;
      return FileIO::Read(f, _roverAndFailure);
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    ostream & RoverAlert::Write(ostream & os) const
    {
      return StreamIO::Write(os, _roverAndFailure);
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    ssize_t RoverAlert::Write(int fd) const
    {
      return DescriptorIO::Write(fd, _roverAndFailure);
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    size_t RoverAlert::Write(FILE *f) const
    {
      return FileIO::Write(f, _roverAndFailure);
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint64_t RoverAlert::StreamedLength() const
    {
      return (IOUtils::StreamedLength(_roverAndFailure));
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool RoverAlert::operator == (const RoverAlert & alert) const
    {
      return (_roverAndFailure == alert._roverAndFailure);
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    string RoverAlert::FailureName() const
    {
      static const map<uint8_t,string>  failureNames = {
        { e_failUnknown,   "unknown" },
        { e_failSocket,    "socket"  },
        { e_failConnect,   "connect" },
        { e_failAuth,      "auth"    },
        { e_failWrite,     "write"   },
        { e_failRead,      "read"    },
        { e_failDecrypt,   "decrypt" }
      };
        
      string  rc = "unknown";
      auto it = failureNames.find(_roverAndFailure.second);
      if (it != failureNames.end()) {
        rc = it->second;
      }
      return rc;
    }
    
    
  }  // namespace Mcrover

}  // namespace Dwm
