//===========================================================================
// @(#) $DwmPath: dwm/mcplex/mcrover/tags/mcrover-0.1.12/classes/src/DwmMcroverUtils.cc 11980 $
// @(#) $Id: DwmMcroverUtils.cc 11980 2022-04-25 02:30:15Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2020
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmMcroverUtils.cc
//!  \author Daniel W. McRobb
//!  \brief Dwm::Mcrover::Utils class implementation
//---------------------------------------------------------------------------

extern "C" {
  #include <sys/types.h>
  #include <sys/socket.h>
  #include <arpa/inet.h>
  #include <netinet/in.h>
  #include <netdb.h>
  #include <unistd.h>
#ifndef __linux__
  #include <dnet.h>
#else
  #include <dumbnet.h>
#endif
}

#include <algorithm>
#include <cstring>  // for memset()

#include "DwmIpv4Prefix.hh"
#include "DwmIpv6Prefix.hh"
#include "DwmSvnTag.hh"
#include "DwmMcroverUtils.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/mcplex/mcrover/tags/mcrover-0.1.12/classes/src/DwmMcroverUtils.cc 11980 $");

namespace Dwm {

  namespace Mcrover {

    using namespace std;
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint32_t Utils::GetHostIpv4Addr(const string & nameOrAddr)
    {
      uint32_t  rc = INADDR_NONE;
      in_addr_t  inAddr = inet_addr(nameOrAddr.c_str());
      if (inAddr != INADDR_NONE) {
        rc = inAddr;
      }
      else {
        struct addrinfo  hints, *addrResult = 0;
        memset(&hints, 0, sizeof(hints));
        hints.ai_family = AF_INET;
        hints.ai_socktype = SOCK_STREAM;
        if (getaddrinfo(nameOrAddr.c_str(), nullptr, &hints,
                        &addrResult) == 0) {
          const struct addrinfo  *addr;
          for (addr = addrResult; addr; addr = addr->ai_next) {
            const struct sockaddr_in  *sain =
              (const struct sockaddr_in *)addr->ai_addr;
            if ((sain->sin_addr.s_addr & inet_addr("127.0.0.0")) !=
                inet_addr("127.0.0.0")) {
              rc = sain->sin_addr.s_addr;
              break;
            }
          }
          freeaddrinfo(addrResult);
        }
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    Ipv6Address Utils::GetHostIpv6Addr(const string & nameOrAddr)
    {
      Ipv6Address  rc;
      in6_addr  in6Addr;
      if (inet_pton(AF_INET6, nameOrAddr.c_str(), &in6Addr) == 1) {
        rc = Ipv6Address(in6Addr);
      }
      else {
        struct addrinfo  hints, *addrResult = 0;
        memset(&hints, 0, sizeof(hints));
        hints.ai_family = AF_INET6;
        hints.ai_socktype = SOCK_STREAM;
        if (getaddrinfo(nameOrAddr.c_str(), nullptr, &hints,
                        &addrResult) == 0) {
          const struct addrinfo  *addr;
          for (addr = addrResult; addr; addr = addr->ai_next) {
            const struct sockaddr_in6  *sain =
              (const struct sockaddr_in6 *)addr->ai_addr;
            if (Ipv6Address(sain->sin6_addr) != Ipv6Address(in6addr_any)) {
              rc = Ipv6Address(sain->sin6_addr);
              break;
            }
          }
          freeaddrinfo(addrResult);
        }
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    string Utils::GetHostName(uint32_t addr)
    {
      string  rc;
      struct sockaddr_in  sockAddr;
      memset(&sockAddr, 0, sizeof(sockAddr));
      sockAddr.sin_family = PF_INET;
      sockAddr.sin_addr.s_addr = addr;
#ifndef __linux__
      sockAddr.sin_len = sizeof(sockAddr);
#endif
      char hbuf[NI_MAXHOST];
      if (getnameinfo((const struct sockaddr *)&sockAddr,
                      sizeof(sockAddr), hbuf, sizeof(hbuf), nullptr, 0,
                      NI_NAMEREQD) == 0) {
        rc = hbuf;
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    std::string Utils::GetHostName(const Ipv6Address & addr)
    {
      string  rc;
      struct sockaddr_in6  sockAddr;
      memset(&sockAddr, 0, sizeof(sockAddr));
      sockAddr.sin6_family = PF_INET6;
      sockAddr.sin6_addr = (struct in6_addr)addr;
#ifndef __linux__
      sockAddr.sin6_len = sizeof(sockAddr);
#endif
      char hbuf[NI_MAXHOST];
      if (getnameinfo((const struct sockaddr *)&sockAddr,
                      sizeof(sockAddr), hbuf, sizeof(hbuf), nullptr, 0,
                      NI_NAMEREQD) == 0) {
        rc = hbuf;
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    string Utils::GetHostName(const IpAddress & addr)
    {
      if (addr.Family() == AF_INET) {
        return GetHostName(addr.Addr<Ipv4Address>()->Raw());
      }
      else {
        return GetHostName(*(addr.Addr<Ipv6Address>()));
      }
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    string Utils::ThisHostName()
    {
      string  rc;
      char  tmp[256] = { '\0' };
      if (gethostname(tmp, sizeof(tmp) - 1) == 0) {
        rc = tmp;
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    IpAddress Utils::ThisHostAddr()
    {
      Ipv6Address  v6Addr = ThisHostIpv6Addr();
      if (Ipv6Address(in6addr_any) != v6Addr) {
        return IpAddress(v6Addr);
      }
      return IpAddress(Ipv4Address(ThisHostIpv4Addr()));
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint32_t Utils::ThisHostIpv4Addr()
    {
      uint32_t  rc = INADDR_NONE;
      string  name = ThisHostName();
      if (! name.empty()) {
        struct addrinfo  hints, *addrResult = 0;
        memset(&hints, 0, sizeof(hints));
        hints.ai_family = AF_INET;
        hints.ai_socktype = SOCK_STREAM;
        if (getaddrinfo(name.c_str(), nullptr, &hints, &addrResult) == 0) {
          const struct addrinfo  *addr;
          for (addr = addrResult; addr; addr = addr->ai_next) {
            const struct sockaddr_in  *sain =
              (const struct sockaddr_in *)addr->ai_addr;
            if ((sain->sin_addr.s_addr & inet_addr("127.0.0.0")) !=
                inet_addr("127.0.0.0")) {
              rc = sain->sin_addr.s_addr;
              break;
            }
          }
          freeaddrinfo(addrResult);
        }
      }
      if (INADDR_NONE == rc) {
        //  host lookup failed.  Use interface address.
        vector<LocalNetInterface>  intfs;
        LocalNetInterface          intfChoice;
        ThisHostInterfaces(intfs);
        for (const auto & intf : intfs) {
          if ((! intf.IsLoopback())) {
            if (intfChoice.Ipv4Addr().Raw() == INADDR_NONE) {
              //  We haven't chosen an interface yet.  This one looks
              //  usable.
              intfChoice = intf;
            }
            else if ((! intfChoice.IsUp()) && intf.IsUp()) {
              //  Original choice was down, but this one's up.  It's a
              //  better choice.
              intfChoice = intf;
            }
          }
        }
        rc = intfChoice.Ipv4Addr().Raw();
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    Ipv6Address Utils::ThisHostIpv6Addr()
    {
      Ipv6Address  rc;
      string  name = ThisHostName();
      if (! name.empty()) {
        struct addrinfo  hints, *addrResult = 0;
        memset(&hints, 0, sizeof(hints));
        hints.ai_family = AF_INET6;
        hints.ai_socktype = SOCK_STREAM;
        if (getaddrinfo(name.c_str(), nullptr, &hints, &addrResult) == 0) {
          const struct addrinfo  *addr;
          for (addr = addrResult; addr; addr = addr->ai_next) {
            const struct sockaddr_in6  *sain =
              (const struct sockaddr_in6 *)addr->ai_addr;
            if (Ipv6Address(sain->sin6_addr) != Ipv6Address(in6addr_any)) {
              rc = Ipv6Address(sain->sin6_addr);
              break;
            }
          }
          freeaddrinfo(addrResult);
        }
      }
      if (Ipv6Address(in6addr_any) == rc) {
        //  host lookup failed.  Use interface address.
        auto sortPref = [] (const auto & a, const auto & b)
                        { bool  lt = false;
                          if (a.IsUp() && (! b.IsUp())) {
                            lt = true;
                          }
                          else {
                            if (a.IsUp() == b.IsUp()) {
                              if (IsLinkLocalAddress(a.Ipv6Addr())
                                  && (! IsLinkLocalAddress(b.Ipv6Addr()))) {
                                lt = true;
                              }
                            }
                          }
                          return lt;
                        };
        vector<LocalNetInterface>  intfs;
        LocalNetInterface          intfChoice;
        ThisHostInterfaces(intfs);
        std::sort(intfs.begin(), intfs.end(), sortPref);
        
        for (const auto & intf : intfs) {
          if ((! intf.IsLoopback())) {
            intfChoice = intf;
            break;
          }
        }
        rc = intfChoice.Ipv6Addr();
      }
      return rc;
    }
        
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint32_t Utils::ThisHostDefaultIpv4Route()
    {
      uint32_t  rc = INADDR_NONE;
      auto  rt = route_open();
      if (rt) {
        route_entry  entry;
        memset(&entry, 0, sizeof(entry));
        entry.route_dst.addr_type = ADDR_TYPE_IP;
        if (route_get(rt, &entry) == 0) {
          if (ADDR_TYPE_IP == entry.route_gw.addr_type) {
            rc = entry.route_gw.addr_ip;
          }
        }
        route_close(rt);
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    static int DnetRouteLoopGetIpv4Routes(const route_entry *entry, void *arg)
    {
      if (entry) {
        if (ADDR_TYPE_IP == entry->route_dst.addr_type) {
          Ipv4Prefix  dstPrefix(entry->route_dst.addr_ip,
                                entry->route_dst.addr_bits);
          if (ADDR_TYPE_IP == entry->route_gw.addr_type) {
            map<Ipv4Prefix,Ipv4Address>  *routes =
              (map<Ipv4Prefix,Ipv4Address> *)arg;
            (*routes)[dstPrefix] = Ipv4Address(entry->route_gw.addr_ip);
          }
          else {
            cerr << "addr_type of route to " << dstPrefix << ": "
                 << entry->route_gw.addr_type << '\n';
          }
        }
      }
      return 0;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Utils::ThisHostIpv4Routes(map<Ipv4Prefix,Ipv4Address> & routes)
    {
      routes.clear();
      auto  rt = route_open();
      if (rt) {
        route_loop(rt, DnetRouteLoopGetIpv4Routes, &routes);
        route_close(rt);
      }
      vector<LocalNetInterface>  intfs;
      if (ThisHostInterfaces(intfs)) {
        for (const auto & intf : intfs) {
          routes[intf.Ipv4Network()] = intf.Ipv4Addr();
        }
      }
      return (! routes.empty());
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Utils::ThisHostsRouteTo(const Ipv4Prefix & prefix, Ipv4Address & gw)
    {
      bool  rc = false;
      map<Ipv4Prefix,Ipv4Address>  routes;
      if (ThisHostIpv4Routes(routes)) {
        for (const auto & route : routes) {
          if (route.first.Contains(prefix)) {
            gw = route.second;
            rc = true;
            break;
          }
        }
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Utils::IsPrivateIpv4Addr(uint32_t addr)
    {
      bool  rc = false;
      static const Ipv4Prefix  privatePrefixes[3] = {
        Ipv4Prefix("10/8"),
        Ipv4Prefix("172.16/12"),
        Ipv4Prefix("192.168/16")
      };
      Ipv4Address ipAddr(addr);
      for (int i = 0; i < 3; ++i) {
        if (privatePrefixes[i].Contains(ipAddr)) {
          rc = true;
          break;
        }
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    static int DnetIntfLoopHandler(const struct intf_entry *entry, void *arg)
    {
      if (entry) {
        if ((ADDR_TYPE_IP == entry->intf_addr.addr_type)
            || (ADDR_TYPE_IP6 == entry->intf_addr.addr_type)) {
          vector<LocalNetInterface>  *intfs =
            (vector<LocalNetInterface> *)arg;
          intfs->push_back(entry);
        }
      }
      return 0;
    }
        
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Utils::ThisHostInterfaces(vector<LocalNetInterface> & intfs)
    {
      intfs.clear();
      auto  intf = intf_open();
      if (intf) {
        intf_loop(intf, DnetIntfLoopHandler, &intfs);
        intf_close(intf);
      }
      return (! intfs.empty());
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Utils::IsLocalAddress(const Ipv4Address & addr)
    {
      bool  rc = false;
      vector<LocalNetInterface>  intfs;
      if (ThisHostInterfaces(intfs)) {
        for (const auto & intf : intfs) {
          if (intf.Ipv4Network().Contains(addr)) {
            rc = true;
            break;
          }
        }
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Utils::IsLinkLocalAddress(const Ipv6Address & addr)
    {
      Ipv6Prefix  linkLocalPrefix("fe80::/10");
      return linkLocalPrefix.Contains(addr);
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    static int DnetArpLoopGetEntries(const struct arp_entry *entry, void *arg)
    {
      if (entry->arp_pa.addr_type == ADDR_TYPE_IP) {
        map<Ipv4Address,MacAddress>  *rc =
          (map<Ipv4Address,MacAddress> *)arg;
        (*rc)[Ipv4Address(entry->arp_pa.addr_ip)] =
          MacAddress(addr_ntoa(&entry->arp_ha));
      }
      return 0;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Utils::ThisHostArpEntries(std::map<Ipv4Address,MacAddress> & entries)
    {
      entries.clear();
      auto  arp = arp_open();
      if (arp) {
        arp_loop(arp, DnetArpLoopGetEntries, &entries);
        arp_close(arp);
      }
      return (! entries.empty());
    }
        
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    static int DnetArpLoopHaveEntry(const struct arp_entry *entry, void *arg)
    {
      pair<Ipv4Address,bool>  *rc = (pair<Ipv4Address,bool> *)arg;
      if (entry->arp_pa.addr_type == ADDR_TYPE_IP) {
        if (rc->first == Ipv4Address(entry->arp_pa.addr_ip)) {
          rc->second = true;
        }
      }
      return 0;
    }
      
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Utils::HaveArpEntry(const Ipv4Address & addr)
    {
      bool  rc = false;
      auto  arp = arp_open();
      if (arp) {
        pair<Ipv4Address,bool>  result(addr, false);
        arp_loop(arp, DnetArpLoopHaveEntry, &result);
        rc = result.second;
        arp_close(arp);
      }
      return rc;
    }
    
    
  }  // namespace Mcrover

}  // namespace Dwm
