//===========================================================================
// @(#) $DwmPath: dwm/mcplex/mcrover/tags/mcrover-0.1.1/classes/src/DwmMcroverDNSUtils.cc 11402 $
// @(#) $Id: DwmMcroverDNSUtils.cc 11402 2020-12-10 08:54:15Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2020
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmMcroverDNSUtils.cc
//!  \author Daniel W. McRobb
//!  \brief Dwm::Mcrover::DNSUtils class implementation
//---------------------------------------------------------------------------

#include <iostream>
#include <map>

#include "DwmSvnTag.hh"
#include "DwmSysLogger.hh"
#include "DwmMcroverNameServerData.hh"
#include "DwmDnsUtils.hh"
#include "DwmMcroverDNSUtils.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/mcplex/mcrover/tags/mcrover-0.1.1/classes/src/DwmMcroverDNSUtils.cc 11402 $");

namespace Dwm {

  namespace Mcrover {

    using namespace std;

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    static void
    PopulateQuestions(const vector<TargetHostConfig> & hosts,
                      map<Ipv4Address,unique_ptr<NameServerData>> *nsdata)
    {
      for (const auto & thc : hosts) {
        if (! thc.DNSResolve().empty()) {
          (*nsdata)[thc.Address()] =
            make_unique<NameServerData>(thc.Address());
          for (const auto & question : thc.DNSResolve()) {
            (*nsdata)[thc.Address()]->Questions().push_back(question);
          }
        }
      }
      return;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    static void
    PerformQueries(map<Ipv4Address,unique_ptr<NameServerData>> *nsdata)
    {
      if (! nsdata->empty()) {
        uint32_t  numQuestions = 0;
        for (auto & ns : *nsdata) {
          numQuestions += ns.second->Questions().size();
          ns.second->SendQuestions();
        }
        uint32_t  numAnswers = 0;
        time_t    now = time((time_t *)0);
        time_t    timeOut = now + 5;
        while ((now < timeOut) && (numAnswers < numQuestions)) {
          for (auto nsit = nsdata->begin(); nsit != nsdata->end(); ) {
            uint32_t  count = nsit->second->ReceiveAnswers();
            if (count) {
              numAnswers += count;
              if (nsit->second->Questions().empty()) {
                nsit = nsdata->erase(nsit++);
                continue;
              }
            }
            ++nsit;
          }
          now = time((time_t *)0);
        }
      }
      return;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    static void
    PopulateAlerts(const AlertOrigin & origin,
                   const map<Ipv4Address,unique_ptr<NameServerData>> *nsdata,
                   AlertBowl & alerts)
    {
      time_t  now = time((time_t *)0);
      for (auto & ns : *nsdata) {
        for (auto & q : ns.second->Questions()) {
          string  qname = q.QName();
          if (q.QType() == Dns::MessageQuestion::k_typePTR) {
            Dns::FromArpa(q.QName(), qname);
          }
          DNSAlert  alert(ns.first, qname, q.QType());
          alerts.Add(origin, alert, now);
        }
      }
      return;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool DNSUtils::GetAlerts(const AlertOrigin & origin,
                             const vector<TargetHostConfig> & hosts,
                             AlertBowl & alerts, bool clearAtEntry)
    {
      if (clearAtEntry) {
        alerts.Clear();
      }
      bool  rc = false;

      //  Populate the questions we'd like to ask.
      map<Ipv4Address,unique_ptr<NameServerData>>  nsData;
      PopulateQuestions(hosts, &nsData);
      
      if (! nsData.empty()) {
        //  Send our questions and receive our answers.
        PerformQueries(&nsData);
        
        //  Try again for the nameservers wth remaining questions to
        //  be answered.
        if (! nsData.empty()) {
          PerformQueries(&nsData);
          if (! nsData.empty()) {
            PopulateAlerts(origin, &nsData, alerts);
          }
        }
      }
      return (! alerts.Empty());
    }
    
  }  // namespace Mcrover

}  // namespace Dwm
