//===========================================================================
// @(#) $DwmPath: dwm/mcplex/mcrover/tags/mcrover-0.1.1/classes/src/DwmMcroverPackMemberConfig.cc 12177 $
// @(#) $Id: DwmMcroverPackMemberConfig.cc 12177 2023-06-20 06:08:17Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2020, 2023
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmMcroverPackMemberConfig.cc
//!  \author Daniel W. McRobb
//!  \brief Dwm::Mcrover::PackMemberConfig class implementation
//---------------------------------------------------------------------------

#include "DwmMcroverPackMemberConfig.hh"

namespace Dwm {

  namespace Mcrover {

    using namespace std;
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    PackMemberConfig::PackMemberConfig()
        : _name(), _addresses()
    {}
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const string & PackMemberConfig::Name() const
    {
      return _name;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const string & PackMemberConfig::Name(const string & name)
    {
      _name = name;
      return _name;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const vector<boost::asio::ip::tcp::endpoint> &
    PackMemberConfig::Addresses() const
    {
      return _addresses;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const vector<boost::asio::ip::tcp::endpoint> &
    PackMemberConfig::Addresses(const vector<boost::asio::ip::tcp::endpoint> & addrs)
    {
      _addresses = addrs;
      return _addresses;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const vector<boost::asio::ip::tcp::endpoint> &
    PackMemberConfig::Addresses(const set<boost::asio::ip::tcp::endpoint> & addrs)
    {
      _addresses.clear();
      for (const auto & si : addrs) {
        _addresses.push_back(si);
      }
      sort(_addresses.begin(), _addresses.end(),
           [] (const auto & a1, const auto & a2)
           {
             if (a1.address().is_v6() && a2.address().is_v4()) {
               return true;
             }
             else if (a1.address().is_v6() == a2.address().is_v6()) {
               return (a1.address() < a2.address());
             }
             return false;
           });
      return _addresses;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool
    PackMemberConfig::AddAddress(const boost::asio::ip::tcp::endpoint & addr)
    {
      bool  rc = false;
      auto  it = find(_addresses.begin(), _addresses.end(), addr);
      if (it == _addresses.end()) {
        _addresses.push_back(addr);
        sort(_addresses.begin(), _addresses.end(),
             [] (const auto & a1, const auto & a2)
             {
               if (a1.address().is_v6() && a2.address().is_v4()) {
                 return true;
               }
               else if (a1.address().is_v6() == a2.address().is_v6()) {
                 return (a1.address() < a2.address());
               }
               return false;
             });
        rc = true;
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    nlohmann::json PackMemberConfig::ToJson() const
    {
      nlohmann::json  j;
      j["name"] = _name;
      for (size_t i = 0; i < _addresses.size(); ++i) {
        nlohmann::json  obj;
        obj["addr"] = _addresses[i].address().to_string();
        obj["port"] = _addresses[i].port();
        j["addresses"][i] = obj;
      }
      return j;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool PackMemberConfig::FromJson(const nlohmann::json & j)
    {
      namespace asip = boost::asio::ip;
      
      bool  rc = false;
      if (j.is_object()) {
        auto  it = j.find("name");
        if ((it != j.end()) && it->is_string()) {
          _name = it->get<string>();
          it = j.find("addresses");
          if ((it != j.end()) && it->is_array()) {
            size_t  e = 0;
            for ( ; e < it->size(); ++e) {
              if ((*it)[e].is_object()) {
                auto  ait = (*it)[e].find("addr");
                if ((ait != (*it)[e].end()) && ait->is_string()) {
                  auto  pit = (*it)[e].find("port");
                  if ((pit != (*it)[e].end()) && pit->is_number()) {
                    asip::tcp::endpoint  ep;
                    ep.address(asip::make_address(ait->get<string>()));
                    ep.port(pit->get<uint16_t>());
                    _addresses.push_back(ep);
                  }
                  else {
                    break;
                  }
                }
                else {
                  break;
                }
              }
              else {
                break;
              }
            }
            rc = (it->size() == e);
          }
        }
      }
      return rc;
    }
    

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    ostream &
    operator << (ostream & os, const PackMemberConfig & cfg) 
    {
      os << "        {\n"
         << "            name = \"" << cfg._name << "\";\n";
      if (! cfg._addresses.empty()) {
        os << "            addresses = [\n";
        auto  iter = cfg._addresses.begin();
        os << "                { address = \"" << iter->address()
           << "\"; port = " << iter->port() << "; }";
        ++iter;
        for ( ; iter != cfg._addresses.end(); ++iter) {
          os << ",\n                { address = \"" << iter->address()
             << "\"; port = " << iter->port() << "; }";
        }
        os << "\n            ];\n";
      }
      os << "        }";
      return os;
    }
    
  }  // namespace Mcrover

}  // namespace Dwm
