//===========================================================================
// @(#) $DwmPath: dwm/mcplex/mcrover/trunk/classes/src/DwmMcroverTcpDstAddr.cc 11949 $
// @(#) $Id: DwmMcroverTcpDstAddr.cc 11949 2022-04-17 17:04:51Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2020
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmMcroverTcpDstAddr.cc
//!  \author Daniel W. McRobb
//!  \brief Dwm::Mcrover::TcpDstAddr class implementation
//---------------------------------------------------------------------------

extern "C" {
  #include <arpa/inet.h>  // for htons() and friends
  #include <unistd.h>
}

#include <cstring>  // for memset()
#include <stdexcept>

#include "DwmDescriptorIO.hh"
#include "DwmFileIO.hh"
#include "DwmIOUtils.hh"
#include "DwmStreamIO.hh"
#include "DwmSvnTag.hh"
#include "DwmMcroverUtils.hh"
#include "DwmMcroverTcpDstAddr.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/mcplex/mcrover/trunk/classes/src/DwmMcroverTcpDstAddr.cc 11949 $");

namespace Dwm {

  namespace Mcrover {

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    TcpDstAddr::TcpDstAddr()
        : _name(), _addr(), _port(0)
    {
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    TcpDstAddr::TcpDstAddr(const std::string & name, IpAddress addr,
                           uint16_t port)
    {
      Set(name, addr, port);
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void TcpDstAddr::Set(const std::string & name, IpAddress addr,
                         uint16_t port)
    {
      _name = name;
      _addr = addr;
      _port = port;
      return;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const std::string & TcpDstAddr::Name() const
    {
      return _name;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const std::string & TcpDstAddr::Name(const std::string & name)
    {
      _name = name;
      return _name;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    IpAddress TcpDstAddr::Address() const
    {
      return _addr;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    IpAddress TcpDstAddr::Address(const IpAddress & addr)
    {
      _addr = addr;
      return _addr;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint16_t TcpDstAddr::Port() const
    {
      return _port;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint16_t TcpDstAddr::Port(uint16_t port)
    {
      return (_port = port);
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool TcpDstAddr::operator < (const TcpDstAddr & dstAddr) const
    {
      bool  rc = false;
      if (_name < dstAddr._name) {
        rc = true;
      }
      else if (_name == dstAddr._name) {
        if (_addr < dstAddr._addr) {
          rc = true;
        }
        else if (_addr == dstAddr._addr) {
          if (_port < dstAddr._port) {
            rc = true;
          }
        }
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    std::istream & TcpDstAddr::Read(std::istream & is)
    {
      if (is) {
        if (StreamIO::Read(is, _name)) {
          if (_addr.Read(is)) {
            StreamIO::Read(is, _port);
          }
        }
      }
      return is;
    }
      
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    ssize_t TcpDstAddr::Read(int fd)
    {
      int  rc = -1;
      if (fd >= 0) {
        int  bytesRead = DescriptorIO::Read(fd, _name);
        if (0 < bytesRead) {
          rc = bytesRead;
          bytesRead = _addr.Read(fd);
          if (bytesRead == IOUtils::StreamedLength(_addr)) {
            rc += bytesRead;
            bytesRead = DescriptorIO::Read(fd, _port);
            if (IOUtils::StreamedLength(_port) == bytesRead) {
              rc += bytesRead;
            }
            else {
              rc = -1;
            }
          }
          else {
            rc = -1;
          }
        }
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    size_t TcpDstAddr::Read(FILE * f)
    {
      size_t  rc = 0;
      
      if (f) {
        if (FileIO::Read(f, _name)) {
          if (FileIO::Read(f, _addr)) {
            if (FileIO::Read(f, _port)) {
              rc = 1;
            }
          }
        }
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    std::ostream & TcpDstAddr::Write(std::ostream & os) const
    {
      if (os) {
        if (StreamIO::Write(os, _name)) {
          if (StreamIO::Write(os, _addr)) {
            StreamIO::Write(os, _port);
          }
        }
      }
      return os;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    ssize_t TcpDstAddr::Write(int fd) const
    {
      ssize_t  rc = -1;
      if (fd >= 0) {
        ssize_t  bytesWritten = DescriptorIO::Write(fd, _name);
        if (bytesWritten > 0) {
          rc = bytesWritten;
          bytesWritten = DescriptorIO::Write(fd, _addr);
          if (bytesWritten > 0) {
            rc += bytesWritten;
            bytesWritten = DescriptorIO::Write(fd, _port);
            if (bytesWritten > 0) {
              rc += bytesWritten;
            }
            else {
              rc = -1;
            }
          }
          else {
            rc = -1;
          }
        }
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    size_t TcpDstAddr::Write(FILE *f) const
    {
      size_t  rc = 0;
      if (f) {
        if (FileIO::Write(f, _name)) {
          if (FileIO::Write(f, _addr)) {
            if (FileIO::Write(f, _port)) {
              rc = 1;
            }
          }
        }
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint64_t TcpDstAddr::StreamedLength() const
    {
      return (IOUtils::StreamedLength(_port)
              + IOUtils::StreamedLength(_addr)
              + IOUtils::StreamedLength(_name));
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    nlohmann::json TcpDstAddr::ToJson() const
    {
      nlohmann::json  j;
      j["name"] = _name;
      j["addr"] = (std::string)_addr;
      j["port"] = _port;
      return j;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool TcpDstAddr::FromJson(const nlohmann::json & json)
    {
      bool  rc = false;
      if (json.is_object()) {
        auto  name = json.find("name");
        if ((name != json.end()) && name->is_string()) {
          _name = name->get<std::string>();
          auto  addr = json.find("addr");
          if ((addr != json.end()) && addr->is_string()) {
            _addr = IpAddress(addr->get<std::string>());
            auto  port = json.find("port");
            if ((port != json.end()) && port->is_number()) {
              _port = port->get<uint16_t>();
              rc = true;
            }
          }
        }
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool TcpDstAddr::operator == (const TcpDstAddr & dst) const
    {
      return ((_name == dst._name)
              && (_addr == dst._addr)
              && (_port == dst._port));
    }

  }  // namespace Mcrover

}  // namespace Dwm
