
//===========================================================================
// @(#) $DwmPath: dwm/mcplex/mcrover/tags/mcrover-0.1.7/apps/mcrover/DwmMcroverAlertsWindow.cc 10997 $
// @(#) $Id: DwmMcroverAlertsWindow.cc 10997 2020-08-27 05:34:16Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2020
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmMcroverAlertsWindow.cc
//!  \author Daniel W. McRobb
//!  \brief NOT YET DOCUMENTED
//---------------------------------------------------------------------------

#include <mutex>
#include <regex>

#include "DwmSvnTag.hh"
#include "DwmMcroverAlertsWindow.hh"
#include "DwmMcroverAlfred1.hh"
#include "DwmMcroverBeastie1.hh"
#include "DwmMcroverCalvin1.hh"
#include "DwmMcroverColorPairs.hh"
#include "DwmMcroverCows1.hh"
#include "DwmMcroverDilbert1.hh"
#include "DwmMcroverDwm1.hh"
#include "DwmMcroverFelix1.hh"
#include "DwmMcroverKermit1.hh"
#include "DwmMcroverSnoopy1.hh"
#include "DwmMcroverSnoopy2.hh"
#include "DwmMcroverSnoopyLucy1.hh"
#include "DwmMcroverTigger1.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/mcplex/mcrover/tags/mcrover-0.1.7/apps/mcrover/DwmMcroverAlertsWindow.cc 10997 $");

namespace Dwm {

  namespace Mcrover {

    using namespace std;

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    static string StripDomains(const std::string & s)
    {
      static const vector<string>  domains = {
        "\\.rfdm\\.com", "\\.mcplex\\.net"
      };
      string  rc = s;
      for (const auto & dom : domains) {
        rc = regex_replace(rc, regex(dom), "");
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    AlertsWindow::AlertsWindow()
    {
      int  h, w;
      getmaxyx(stdscr, h, w);
      _win = newwin(h - 2, w, 1, 0);
      // scrollok(_win, true);
      _contentLength = 0;
      _scrollPos = 0;
      
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    AlertsWindow::~AlertsWindow()
    {
      if (_win) {
        delwin(_win);
        _win = nullptr;
      }
    }
  
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void AlertsWindow::
    Update(const vector<pair<Mcrover::AlertBowl::KeyType,Mcrover::AlertBowlValue>> & alerts)
    {
      wmove(_win, 0, 0);
      if (alerts.empty()) {
        wattrset(_win, COLOR_PAIR(ColorPairs::Pair("white")));
        wclrtobot(_win);
        _contentLength = DrawAsciiArt();
      }
      else {
        _contentLength = alerts.size();
        int  winh, winw;
        getmaxyx(_win, winh, winw);
        for (int n = _scrollPos; n < alerts.size(); ++n) {
          if ((n - _scrollPos) > (winh - 1)) {
            break;
          }
          string  dispstr = alerts[n].second.DisplayString() + ' ';
          std::visit([&dispstr](auto const & p)
          { dispstr += p.DisplayString(); }, alerts[n].first);
          DrawAlertLine(dispstr, n - _scrollPos);
        }
      }
      wclrtobot(_win);
      wrefresh(_win);
      return;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void AlertsWindow::Resize()
    {
      wresize(_win, LINES - 2, COLS);
      mvwin(_win, 1, 0);
      return;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void AlertsWindow::Scroll(int amt)
    {
      int  winh, winw;
      getmaxyx(_win, winh, winw);
      if (_contentLength <= winh) {
        _scrollPos = 0;
      }
      else {
        int  ypos;
        int  newPos = _scrollPos + amt;
        if (newPos < 0) {
          newPos = 0;
        }
        else if (newPos > (_contentLength - 1)) {
          newPos = (_contentLength - 1);
          if (newPos < 0) {
            newPos = 0;
          }
        }
        _scrollPos = newPos;
      }
      return;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void AlertsWindow::ScrollPage(int pages)
    {
      int  winh, winw;
      getmaxyx(_win, winh, winw);
      Scroll(pages * winh);
      return;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    int AlertsWindow::DrawAsciiArt()
    {
      int             rc = 0;
      static uint8_t  pic = 0;
      static vector<unique_ptr<AsciiArt>>  art;
      static mutex    artmtx;
      {
        lock_guard<mutex>  artlock(artmtx);
        if (art.empty()) {
          art.push_back(make_unique<Snoopy1>());
          art.push_back(make_unique<Dwm1>());
          art.push_back(make_unique<Snoopy2>());
          art.push_back(make_unique<SnoopyLucy1>());
          art.push_back(make_unique<Dilbert1>());
          art.push_back(make_unique<Beastie1>());
          art.push_back(make_unique<Cows1>());
          art.push_back(make_unique<Kermit1>());
          art.push_back(make_unique<Felix1>());
          art.push_back(make_unique<Tigger1>());
          art.push_back(make_unique<Calvin1>());
          art.push_back(make_unique<Alfred1>());
        }
        rc = art[pic]->Height();
        art[pic]->Draw(_win);
        ++pic;
        if (pic >= art.size()) {
          pic = 0;
        }
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void AlertsWindow::DrawAlertLine(const string & alert, int line)
    {
      static const  int cyan = ColorPairs::Pair("cyan");
      static const  int white = ColorPairs::Pair("white");
      int               violet, yellowgreen;
      if (16 > COLORS) {
        violet = ColorPairs::Pair("magenta");
        yellowgreen = ColorPairs::Pair("green");
      }
      else {
        violet = ColorPairs::Pair("violet");
        yellowgreen = ColorPairs::Pair("brightgreen");
      }
      
      static const regex
        rgx("([0-9][0-9][/][0-9][0-9] [0-9][0-9][:][0-9][0-9]) ([^ ]+ )([^ ]+ )(.+)",
            regex::ECMAScript|regex::optimize);
      smatch  sm;
      string  alertstr = StripDomains(alert);
      if (regex_match(alertstr, sm, rgx)) {
        if (sm.size() == 5) {
          wmove(_win, line, 0);
          wclrtoeol(_win);
          wmove(_win, line, 0);
          wattrset(_win, COLOR_PAIR(cyan)|A_NORMAL);
          waddstr(_win, sm[1].str().c_str());
          wattrset(_win, COLOR_PAIR(violet)|A_BOLD);
          wmove(_win, line, 12);
          waddstr(_win, sm[2].str().c_str());
          wattrset(_win, COLOR_PAIR(white)|A_BOLD);
          waddstr(_win, sm[3].str().c_str());
          wattrset(_win, COLOR_PAIR(yellowgreen)|A_BOLD);
          waddstr(_win, sm[4].str().c_str());
        }
      }
      else {
        wmove(_win, line, 0);
        wclrtoeol(_win);
        wmove(_win, line, 0);
        wattrset(_win, COLOR_PAIR(white)|A_BOLD);
        waddstr(_win, alertstr.c_str());
      }
      return;
    }
    
  }  // namespace Mcrover
  
}  // namespace Dwm
