//===========================================================================
// @(#) $Name:$
// @(#) $Id: DwmMcroverSlidingStackedWidget.cc 10997 2020-08-27 05:34:16Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2020
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmMcroverSlidingStackedWidget.cc
//!  \brief Dwm::Mcrover::SlidingStackedWidget class implementation
//---------------------------------------------------------------------------

#include "DwmSvnTag.hh"
#include "DwmMcroverSlidingStackedWidget.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/mcplex/mcrover/tags/mcrover-0.1.7/apps/qmcrover/DwmMcroverSlidingStackedWidget.cc 10997 $");

namespace Dwm {

  namespace Mcrover {

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    SlidingStackedWidget::SlidingStackedWidget(QWidget *parent)
        : QStackedWidget(parent)
    {
      _vertical      = false;
      _duration      = 750;
      _now           = 0;
      _next          = 0;
      _wrap          = false;
      _pnow          = QPoint(0,0);
      _active        = false;
      _animationType = QEasingCurve::InOutCubic;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    SlidingStackedWidget::~SlidingStackedWidget()
    {
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void SlidingStackedWidget::SetVerticalMode(bool vertical)
    {
      _vertical = vertical;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void SlidingStackedWidget::SetDuration(int msecs)
    {
      _duration = msecs;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void
    SlidingStackedWidget::SetAnimation(enum QEasingCurve::Type animationtype)
    {
      _animationType = animationtype;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void SlidingStackedWidget::SetWrap(bool wrap)
    {
      _wrap = wrap;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void SlidingStackedWidget::SlideInNext()
    {
      int now = currentIndex();
      if (_wrap || (now < count() - 1))
        SlideInIdx(now + 1);
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void SlidingStackedWidget::SlideInPrev()
    {
      int now = currentIndex();
      if (_wrap || (now > 0))
        SlideInIdx(now - 1);
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void SlidingStackedWidget::SlideInIdx(int idx, AnimDirection direction)
    {
      if (idx > (count() - 1)) {
        direction = _vertical ? e_topToBottom : e_rightToLeft;
        idx = idx % count();
      }
      else if (idx < 0) {
        direction = _vertical ? e_bottomToTop: e_leftToRight;
        idx = (idx + count()) % count();
      }
      SlideInWgt(widget(idx), direction);
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void SlidingStackedWidget::SlideInWgt(QWidget *newwidget,
                                          AnimDirection direction)
    {
      if (_active) {
        //  do not allow re-entrance before an animation is complete
        return;
      }
      else {
        _active = true;
      }

      AnimDirection  directionHint;
      int now  = currentIndex();
      int next = indexOf(newwidget);
      if (now == next) {
        _active = false;
        return;
      }
      else if (now < next) {
        directionHint = _vertical ? e_topToBottom : e_rightToLeft;
      }
      else {
        directionHint = _vertical ? e_bottomToTop : e_leftToRight;
      }
      if (direction == e_automatic) {
        direction = directionHint;
      }
      int offsetx = frameRect().width();
      int offsety = frameRect().height();

      // The following is important, to ensure that the new widget
      // has correct geometry information when sliding in first time
      widget(next)->setGeometry (0, 0, offsetx, offsety);

      if (direction == e_bottomToTop)  {
        offsetx = 0;
        offsety = -offsety;
      }
      else if (direction == e_topToBottom) {
        offsetx = 0;
      }
      else if (direction == e_rightToLeft) {
        offsetx = -offsetx;
        offsety = 0;
      }
      else if (direction == e_leftToRight) {
        offsety = 0;
      }
      // Re-position the next widget outside/aside of the display area
      QPoint pnext = widget(next)->pos();
      QPoint pnow  = widget(now)->pos();
      _pnow = pnow;

      widget(next)->move(pnext.x() - offsetx, pnext.y() - offsety);
      // Make it visible/show
      widget(next)->show();
      widget(next)->raise();

      //  Animate both, the now and next widget to the side, using
      //  animation framework.
      QPropertyAnimation *animnow =
        new QPropertyAnimation(widget(now), "pos");
      animnow->setDuration(_duration);
      animnow->setEasingCurve(_animationType);
      animnow->setStartValue(QPoint(pnow.x(), pnow.y()));
      animnow->setEndValue(QPoint(offsetx + pnow.x(), offsety + pnow.y()));
      QPropertyAnimation *animnext =
        new QPropertyAnimation(widget(next), "pos");
      animnext->setDuration(_duration);
      animnext->setEasingCurve(_animationType);
      animnext->setStartValue(QPoint(-offsetx + pnext.x(), offsety + pnext.y()));
      animnext->setEndValue(QPoint(pnext.x(), pnext.y()));

      QParallelAnimationGroup *animgroup = new QParallelAnimationGroup;

      animgroup->addAnimation(animnow);
      animgroup->addAnimation(animnext);

      QObject::connect(animgroup, SIGNAL(finished()), this,
                       SLOT(AnimationDoneSlot()));
      _next   = next;
      _now    = now;
      _active = true;
      animgroup->start();
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void SlidingStackedWidget::AnimationDoneSlot(void)
    {
      setCurrentIndex(_next);
      // Then hide the outshifted widget now
      widget(_now)->hide();
      // Then set the position of the outshifted widget now back to
      // its original position
      widget(_now)->move(_pnow);
      
      _active = false;
      emit AnimationFinished();
    }

    
  }  // namespace Mcrover

}  // namespace Dwm
