//===========================================================================
// @(#) $DwmPath: dwm/mcplex/mcrover/tags/mcrover-0.1.7/classes/include/DwmMcroverSMTPStateMachine.hh 11980 $
// @(#) $Id: DwmMcroverSMTPStateMachine.hh 11980 2022-04-25 02:30:15Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2020
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmMcroverSMTPStateMachine.hh
//!  \author Daniel W. McRobb
//!  \brief Dwm::Mcrover::SMTPStateMachine class declaration, and declaration
//!         of supporting classes
//---------------------------------------------------------------------------

//----------------------------------------------------------------------------
//  NOTES
//  For each target,  I need to:
//   - try to connect (non-blocking)
//   - read the initial response
//   - send a NOOP
//   - read NOOP response (non-blocking)
//   - close connection (or time out)
//----------------------------------------------------------------------------

#ifndef _DWMMCROVERSMTPSTATEMACHINE_HH_
#define _DWMMCROVERSMTPSTATEMACHINE_HH_

#include <cstdint>
#include <mutex>
#include <string>

#include "DwmIpAddress.hh"

namespace Dwm {

  namespace Mcrover {

    class SMTPStateMachine;
    
    //------------------------------------------------------------------------
    //!  Base class for all SMTP states.
    //------------------------------------------------------------------------
    class SMTPState
    {
    public:
      virtual void Start(SMTPStateMachine *sm);
      virtual void Continue(SMTPStateMachine *sm);
      virtual void Stop(SMTPStateMachine *sm);
      virtual std::string Name() const;
    };

    //------------------------------------------------------------------------
    //!  The initial state.
    //------------------------------------------------------------------------
    class SMTPInitial
      : public SMTPState
    {
    public:
      static SMTPInitial *Instance();
      void Start(SMTPStateMachine *sm) override;
      void Continue(SMTPStateMachine *sm) override;
      void Stop(SMTPStateMachine *sm) override;
      std::string Name() const override   { return "Initial"; }
    protected:
      SMTPInitial() = default;
    private:
      static std::mutex    _instanceMutex;
      static SMTPInitial  *_instance;
    };
    
    //------------------------------------------------------------------------
    //!  The connecting state.
    //------------------------------------------------------------------------
    class SMTPConnecting
      : public SMTPState
    {
    public:
      static SMTPConnecting *Instance();
      void Continue(SMTPStateMachine *sm) override;
      void Stop(SMTPStateMachine *sm) override;
      std::string Name() const override   { return "Connecting"; }
    protected:
      SMTPConnecting() = default;
    private:
      static std::mutex       _instanceMutex;
      static SMTPConnecting  *_instance;
    };
    
    //------------------------------------------------------------------------
    //!  The 'awaiting initial response' state.
    //------------------------------------------------------------------------
    class SMTPAwaitingInitialResponse
      : public SMTPState
    {
    public:
      static SMTPAwaitingInitialResponse *Instance();
      void Continue(SMTPStateMachine *sm) override;
      void Stop(SMTPStateMachine *sm) override;
      std::string Name() const override   { return "AwaitingInitialResponse"; }
    protected:
      SMTPAwaitingInitialResponse() = default;
    private:
      static std::mutex                    _instanceMutex;
      static SMTPAwaitingInitialResponse  *_instance;
    };

    //------------------------------------------------------------------------
    //!  The 'sending no-op' state.
    //------------------------------------------------------------------------
    class SMTPSendingNoOp
      : public SMTPState
    {
    public:
      static SMTPSendingNoOp *Instance();
      void Continue(SMTPStateMachine *sm) override;
      void Stop(SMTPStateMachine *sm) override;
      std::string Name() const override   { return "SendingNoOp"; }
    protected:
      SMTPSendingNoOp() = default;
    private:
      static std::mutex        _instanceMutex;
      static SMTPSendingNoOp  *_instance;
    };
    
    //------------------------------------------------------------------------
    //!  The 'awaiting no-op response' state.
    //------------------------------------------------------------------------
    class SMTPAwaitingNoOpResponse
      : public SMTPState
    {
    public:
      static SMTPAwaitingNoOpResponse *Instance();
      void Continue(SMTPStateMachine *sm) override;
      void Stop(SMTPStateMachine *sm) override;
      std::string Name() const override   { return "AwaitingNoOpResponse"; }
    protected:
      SMTPAwaitingNoOpResponse() = default;
    private:
      static std::mutex                 _instanceMutex;
      static SMTPAwaitingNoOpResponse  *_instance;
    };
    
    //------------------------------------------------------------------------
    //!  The 'stopped' state.
    //------------------------------------------------------------------------
    class SMTPStopped
      : public SMTPState
    {
    public:
      static SMTPStopped *Instance();
      std::string Name() const override   { return "Stopped"; }
    protected:
      SMTPStopped() = default;
    private:
      static std::mutex    _instanceMutex;
      static SMTPStopped  *_instance;
    };

    //------------------------------------------------------------------------
    //!  The 'failed' state.
    //------------------------------------------------------------------------
    class SMTPFailed
      : public SMTPState
    {
    public:
      static SMTPFailed *Instance();
      std::string Name() const override   { return "Failed"; }
    protected:
      SMTPFailed() = default;
    private:
      static std::mutex   _instanceMutex;
      static SMTPFailed  *_instance;
    };

    //------------------------------------------------------------------------
    //!  The 'passed' state.
    //------------------------------------------------------------------------
    class SMTPPassed
      : public SMTPState
    {
    public:
      static SMTPPassed *Instance();
      std::string Name() const override   { return "Passed"; }
    protected:
      SMTPPassed() = default;
    private:
      static std::mutex   _instanceMutex;
      static SMTPPassed  *_instance;
    };
    
    //------------------------------------------------------------------------
    //!  Encapsulates an SMTP state machine.
    //------------------------------------------------------------------------
    class SMTPStateMachine
    {
    public:
      using Clock = std::chrono::system_clock;
      
      SMTPStateMachine(const IpAddress & addr, uint16_t port);
      
      bool Start();
      bool Continue();
      bool Stop();
      void Reset();
      
      bool Done() const;

      bool OpenSocket();
      bool Connect();
      void CloseSocket();
      void ChangeState(SMTPState *state);
      
      int Fd() const                           { return _fd; }
      SMTPState *CurrentState()                { return _currentState; }
      SMTPState *PreviousState()               { return _previousState; }
      std::string & InitialResponse()          { return _initialResponse; }
      static const std::string & NoOpString()  { return _noOpString; }
      size_t & NoOpBytesSent()                 { return _noOpBytesSent; }
      std::string & NoOpResponse()             { return _noOpResponse; }
      Clock::duration TimeInCurrentState() const;
      
    private:
      // Ipv4Address                 _addr;
      IpAddress                   _addr;
      uint16_t                    _port;
      int                         _fd;
      SMTPState                  *_currentState;
      SMTPState                  *_previousState;
      Clock::time_point           _lastStateChangeTime;
      std::string                 _initialResponse;
      size_t                      _noOpBytesSent;
      std::string                 _noOpResponse;
      static const std::string    _noOpString;
    };
    
  }  // namespace Mcrover

}  // namespace Dwm

#endif  // _DWMMCROVERSMTPSTATEMACHINE_HH_
