//===========================================================================
// @(#) $DwmPath$
// @(#) $Id$
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2022
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmMcroverTcpDstAddr.hh
//!  \author Daniel W. McRobb
//!  \brief Dwm::Mcrover::TcpDstAddr class declaration
//---------------------------------------------------------------------------

#ifndef _DWMMCROVERTCPDSTADDR_HH_
#define _DWMMCROVERTCPDSTADDR_HH_

extern "C" {
  #include <sys/types.h>
  #include <sys/socket.h>
  #include <netinet/in.h>
}

#include <cstdint>
#include <string>

#include <nlohmann/json.hpp>

#include "DwmDescriptorIOCapable.hh"
#include "DwmFileIOCapable.hh"
#include "DwmIpAddress.hh"
#include "DwmStreamIOCapable.hh"
#include "DwmStreamedLengthCapable.hh"

namespace Dwm {

  namespace Mcrover {

    //------------------------------------------------------------------------
    //!  Encapsulates a TCP destination (host and port).
    //------------------------------------------------------------------------
    class TcpDstAddr
      : public DescriptorIOCapable, public FileIOCapable,
        public StreamIOCapable, public StreamedLengthCapable
    {
    public:
      TcpDstAddr();
      TcpDstAddr(const TcpDstAddr &) = default;
      TcpDstAddr & operator = (const TcpDstAddr &) = default;

      TcpDstAddr(const std::string & name, IpAddress addr, uint16_t port);
      void Set(const std::string & name, IpAddress addr, uint16_t port);
      
      struct sockaddr SockAddr() const;
      const std::string & Name() const;
      const std::string & Name(const std::string & name);
      IpAddress Address() const;
      IpAddress Address(const IpAddress & addr);
      uint16_t Port() const;
      uint16_t Port(uint16_t port);
      bool operator < (const TcpDstAddr & dstAddr) const;

      std::istream & Read(std::istream & is) override;
      ssize_t Read(int fd) override;
      size_t Read(FILE * f) override;

      std::ostream & Write(std::ostream & os) const override;
      ssize_t Write(int fd) const override;
      size_t Write(FILE *f) const override;
      uint64_t StreamedLength() const override;

      nlohmann::json ToJson() const;
      bool FromJson(const nlohmann::json & json);

      bool operator == (const TcpDstAddr & dst) const;
      
    private:
      std::string      _name;
      IpAddress        _addr;
      uint16_t         _port;
      struct sockaddr  _sockAddr;
    };
    
  }  // namespace Mcrover

}  // namespace Dwm

#endif  // _DWMMCROVERTCP4DSTADDR_HH_
