//===========================================================================
// @(#) $DwmPath: dwm/mcplex/mcrover/tags/mcrover-0.1.7/classes/src/DwmMcroverLocalHostConfig.cc 12283 $
// @(#) $Id: DwmMcroverLocalHostConfig.cc 12283 2023-11-14 05:05:22Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2020, 2023
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmMcroverLocalHostConfig.cc
//!  \author Daniel W. McRobb
//!  \brief Dwm::Mcrover::LocalHostConfig class implementation
//---------------------------------------------------------------------------

#include "DwmSvnTag.hh"
#include "DwmMcroverLocalHostConfig.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/mcplex/mcrover/tags/mcrover-0.1.7/classes/src/DwmMcroverLocalHostConfig.cc 12283 $");

namespace Dwm {

  namespace Mcrover {

    using namespace std;
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    LocalHostConfig::LocalHostConfig()
        : _zpools(), _filesystems(), _routes()
    {}
    
      
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const map<string,uint8_t> & LocalHostConfig::Zpools() const
    {
      return _zpools;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    map<string,uint8_t> & LocalHostConfig::Zpools()
    {
      return _zpools;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const map<string,uint8_t> & LocalHostConfig::Filesystems() const
    {
      return _filesystems;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    map<string,uint8_t> & LocalHostConfig::Filesystems()
    {
      return _filesystems;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const map<string,string> & LocalHostConfig::Disks() const
    {
      return _disks;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    map<string,string> & LocalHostConfig::Disks()
    {
      return _disks;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const map<string,uint64_t> & LocalHostConfig::Dumps() const
    {
      return _dumps;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    map<string,uint64_t> & LocalHostConfig::Dumps()
    {
      return _dumps;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const map<Ipv4Prefix,vector<Ipv4Prefix>> &
    LocalHostConfig::Routes() const
    {
      return _routes;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    map<Ipv4Prefix,vector<Ipv4Prefix>> & LocalHostConfig::Routes()
    {
      return _routes;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const vector<Ipv4Prefix> & LocalHostConfig::Guests() const
    {
      return _guests;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    vector<Ipv4Prefix> & LocalHostConfig::Guests()
    {
      return _guests;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const vector<Ipv4Prefix> & LocalHostConfig::Bandits() const
    {
      return _bandits;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    vector<Ipv4Prefix> & LocalHostConfig::Bandits()
    {
      return _bandits;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    static void
    PrintRoute(ostream & os,
               const pair<Ipv4Prefix,vector<Ipv4Prefix>> & route)
    {
      os << "        {\n";
      if (route.first == Ipv4Prefix("0/0")) {
        os << "            default = [\n";
      }
      else {
        os << "            " << route.first << " = [\n";
      }
      if (! route.second.empty()) {
        auto  iter = route.second.begin();
        os << "                \"" << *iter << '"';
        ++iter;
        for ( ; iter != route.second.end(); ++iter) {
          os << ",\n                \"" << *iter << '"';
        }
        os << "\n            ];\n";
      }
      os << "        }\n";
      return;
    }
      
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    static void
    PrintRoutes(ostream & os,
                const map<Ipv4Prefix,vector<Ipv4Prefix>> & routes)
    {
      if (! routes.empty()) {
        os << "    #---------------------------------------------------------"
          "---------------\n"
           << "    #  \n"
           << "    #---------------------------------------------------------"
          "---------------\n";
        os << "    ROUTES = [\n";
        auto  iter = routes.begin();
        PrintRoute(os, *iter);
        ++iter;
        for ( ; iter != routes.end(); ++iter) {
          os << ",\n";
          PrintRoute(os, *iter);
        }
        os << "    ];\n\n";
      }
    }

    static void
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    PrintPrefixList(ostream & os, const string & name,
                    const vector<Ipv4Prefix> & pfxs)
    {
      if (! pfxs.empty()) {
        os << "    " << name << " = [\n";
        auto  iter = pfxs.begin();
        os << "        \"" << *iter << '"';
        ++iter;
        for ( ; iter != pfxs.end(); ++iter) {
          os << ",\n        \"" << *iter << '"';
        }
        os << "\n    ];\n\n";
      }
      return;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    static void
    PrintGuests(ostream & os, const vector<Ipv4Prefix> & guests)
    {
      if (! guests.empty()) {
        os << "    #---------------------------------------------------------"
          "---------------\n"
           << "    #  I want an alert if a 'guest' address shows up in my ARP"
          " cache.\n"
           << "    #  A 'guest' is any address within one of these configured"
          " prefixes.\n"
           << "    #  Typically it's the pool of IP addresses handed out by"
          " a local\n"
           << "    #  DHCP server.\n"
           << "    #---------------------------------------------------------"
          "---------------\n";
        PrintPrefixList(os, "GUESTS", guests);
      }
      return;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    static void
    PrintBandits(ostream & os, const vector<Ipv4Prefix> & bandits)
    {
      if (! bandits.empty()) {
        os << "    #---------------------------------------------------------"
          "---------------\n"
           << "    #  Alert me if I see an ARP cache entry for any address in"
          " these\n"
           << "    #  prefixes.\n"
           << "    #---------------------------------------------------------"
          "---------------\n";
        PrintPrefixList(os, "BANDITS", bandits);
      }
      return;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    static void PrintDisks(ostream & os,
                           const map<string,string> & disks)
    {
      if (! disks.empty()) {
        os << "    #---------------------------------------------------------"
          "---------------\n"
           << "    #  Check the S.M.A.R.T. health status of disks.\n"
           << "    #---------------------------------------------------------"
          "---------------\n";
        os << "    DISKS = [\n";
        auto  iter = disks.begin();
        os << "        { name = \"" << iter->first << "\"; device = \""
           << iter->second << "\"; }";
        ++iter;
        for (; iter != disks.end(); ++iter) {
          os << ",\n        { name = \"" << iter->first << "\"; device = \""
             << iter->second << "\"; }";
        }
        os << "\n    ];\n\n";
      }
      return;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    static string PeriodString(uint64_t dur)
    {
      string  rc;
      uint64_t  w = dur / (7 * 24 * 60 * 60);
      if (w) {
        rc += to_string(w) + 'w';
        dur -= (w * 7 * 24 * 60 * 60);
      }
      uint64_t  d = dur / (24 * 60 * 60);
      if (d) {
        rc += to_string(d) + 'd';
        dur -= (d * 24 * 60 * 60);
      }
      uint64_t  h = dur / (60 * 60);
      if (h) {
        rc += to_string(h) + 'h';
        dur -= (h * 60 * 60);
      }
      uint64_t  m = dur / 60;
      if (m) {
        rc += to_string(m) + 'm';
        dur -= (m * 60);
      }
      if (dur) {
        rc += to_string(dur) + 's';
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    static void PrintDumps(ostream & os,
                           const map<string,uint64_t> & dumps)
    {
      if (! dumps.empty()) {
        os << "    #---------------------------------------------------------"
          "---------------\n"
           << "    #  Check that a 'dump' of the given devices has occurred"
          " within the\n"
           << "    #  given time interval.\n"
           << "#-------------------------------------------------------------"
           << "-----------\n";
        os << "    DUMPS = [\n";
        auto  iter = dumps.begin();
        os << "        { device = \"" << iter->first << "\"; period = "
           << PeriodString(iter->second) << "; }";
        ++iter;
        for (; iter != dumps.end(); ++iter) {
          os << ",\n        { device = \"" << iter->first << "\"; period = "
             << PeriodString(iter->second) << "; }";
        }
        os << "\n    ];\n\n";
      }
      return;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    static void
    PrintFilesystems(ostream & os,
                     const map<string,uint8_t> & filesystems)
    {
      if (! filesystems.empty()) {
        os << "    #---------------------------------------------------------"
          "---------------\n"
           << "    #  Check that filesystems are mounted, and optionally"
          " check their\n"
           << "    #  capacity.\n"
           << "    #---------------------------------------------------------"
          "---------------\n";
        os << "    FS = [\n";
        auto  iter = filesystems.begin();
        os << "        { mount = \"" << iter->first << '"';
        if (iter->second < 255) {
          os << "; capacity = " << (uint16_t)iter->second;
        }
        os << "; }";
        ++iter;
        for ( ; iter != filesystems.end(); ++iter) {
          os << ",\n        { name = \"" << iter->first << '"';
          if (iter->second < 255) {
            os << "; capacity = " << (uint16_t)iter->second;
          }
          os << "; }";
        }
        os << "\n    ];\n\n";
      }

      return;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    static void PrintZpools(ostream & os,
                            const map<string,uint8_t> & zpools)
    {
      if (! zpools.empty()) {
        os << "    #---------------------------------------------------------"
          "---------------\n"
           << "    #  For any listed ZFS pool, we will check that its status"
          " is ONLINE.\n"
           << "    #  Optionally, we can also check that it's not above the"
          " given capacity\n"
           << "    #  (percentage).\n"
           << "    #---------------------------------------------------------"
          "---------------\n";
        os << "    ZPOOL = [\n";
        auto  iter = zpools.begin();
        os << "        { name = \"" << iter->first << '"';
        if (iter->second < 255) {
          os << "; capacity = " << (uint16_t)iter->second;
        }
        os << "; }";
        ++iter;
        for ( ; iter != zpools.end(); ++iter) {
          os << ",\n        { name = \"" << iter->first << '"';
          if (iter->second < 255) {
            os << "; capacity = " << (uint16_t)iter->second;
          }
          os << "; }";
        }
        os << "\n    ];\n\n";
      }
      return;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    ostream &
    operator << (ostream & os, const LocalHostConfig & cfg)
    {
      os << "#==============================================================="
        "=============\n"
         << "#  Local checks.\n"
         << "#==============================================================="
        "=============\n"
         << "local {\n";

      PrintZpools(os, cfg._zpools);
      PrintFilesystems(os, cfg._filesystems);
      PrintDumps(os, cfg._dumps);
      PrintDisks(os, cfg._disks);
      PrintRoutes(os, cfg._routes);
      PrintGuests(os, cfg._guests);
      PrintBandits(os, cfg._bandits);
      
      os << "};\n";

      return os;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    nlohmann::json LocalHostConfig::ToJson() const
    {
      nlohmann::json  j;
      for (auto zpool : _zpools) {
        j["zpools"][zpool.first] = (uint16_t)zpool.second;
      }
      for (auto fs : _filesystems) {
        j["filesystems"][fs.first] = (uint16_t)fs.second;
      }
      for (auto disk : _disks) {
        j["disks"][disk.first] = disk.second;
      }
      for (auto dump : _dumps) {
        j["dumps"][dump.first] = dump.second;
      }
      for (auto route : _routes) {
        size_t  i = 0;
        for (auto pfx : route.second) {
          j["routes"][route.first.ToString()][i] = pfx.ToString();
          ++i;
        }
      }
      size_t  g = 0;
      for (auto guest : _guests) {
        j["guests"][g] = guest.ToString();
        ++g;
      }
      size_t  b = 0;
      for (auto bandit : _bandits) {
        j["bandits"][b] = bandit.ToString();
        ++b;
      }
      return j;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    static bool FromJsonMap(const nlohmann::json & j, const string & name,
                            map<string,uint8_t> & m)
    {
      bool  rc = false;
      if (j.is_object()) {
        rc = true;
        auto  it = j.find(name);
        if (it != j.end()) {
          if (it->is_object()) {
            auto  oit = it->begin();
            for ( ; oit != it->end(); ++oit) {
              if (oit.value().is_number()) {
                uint16_t  val = oit.value().get<uint16_t>();
                if (val <= 255) {
                  m[oit.key()] = (uint8_t)val;
                }
                else {
                  rc = false;
                  break;
                }
              }
              else {
                rc = false;
                break;
              }
            }
          }
          else {
            rc = false;
          }
        }
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    static bool FromJsonMap(const nlohmann::json & j,
                            map<string,uint64_t> & m)
    {
      bool  rc = false;
      m.clear();
      if (j.is_object()) {
        auto  it = j.begin();
        for ( ; it != j.end(); ++it) {
          if (it.value().is_number()) {
            m[it.key()] = it.value().get<uint64_t>();
          }
          else {
            break;
          }
        }
        rc = (j.end() == it);
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    static bool FromJsonMap(const nlohmann::json & j,
                            map<string,string> & m)
    {
      bool  rc = false;
      m.clear();
      if (j.is_object()) {
        auto  it = j.begin();
        for ( ; it != j.end(); ++it) {
          if (it.value().is_string()) {
            m[it.key()] = it.value().get<string>();
          }
          else {
            break;
          }
        }
        rc = (j.end() == it);
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    static bool FromJsonMap(const nlohmann::json & j,
                            map<Ipv4Prefix,vector<Ipv4Prefix>> & routes)
    {
      bool  rc = false;
      if (j.is_object()) {
        auto  pit = j.begin();
        for ( ; pit != j.end(); ++pit) {
          if (pit.value().is_array()) {
            size_t  i = 0;
            for ( ; i < pit.value().size(); ++i) {
              if (pit.value()[i].is_string()) {
                Ipv4Prefix  newPrefix(pit.value()[i].get<string>());
                routes[Ipv4Prefix(pit.key())].push_back(newPrefix);
              }
              else {
                break;
              }
            }
            rc = (pit.value().size() == i);
            if (! rc) {
              break;
            }
          }
        }
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    static bool FromJsonArray(const nlohmann::json & j,
                              vector<Ipv4Prefix> & pfxs)
    {
      bool  rc = false;
      if (j.is_array()) {
        size_t  i = 0;
        for ( ; i != j.size(); ++i) {
          if (j[i].is_string()) {
            Ipv4Prefix  pfx(j[i].get<string>());
            if ((pfx.MaskLength() != 32)
                && (pfx.Network() != Ipv4Address("255.255.255.255"))) {
              pfxs.push_back(pfx);
            }
            else {
              break;
            }
          }
          else {
            break;
          }
        }
        rc = (j.size() == i);
      }
      return rc;
    }
            
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool LocalHostConfig::FromJson(const nlohmann::json & j)
    {
      bool  rc = true;
      Clear();
      if (j.is_object()) {
        rc &= FromJsonMap(j, "zpools", _zpools);
        rc &= FromJsonMap(j, "filesystems", _filesystems);
        auto  it = j.find("disks");
        if (it != j.end()) {
          rc &= FromJsonMap(*it, _disks);
        }
        it = j.find("dumps");
        if (it != j.end()) {
          rc &= FromJsonMap(*it, _dumps);
        }
        it = j.find("routes");
        if (it != j.end()) {
          rc &= FromJsonMap(*it, _routes);
        }
        it = j.find("guests");
        if (it != j.end()) {
          rc &= FromJsonArray(*it, _guests);
        }
        it = j.find("bandits");
        if (it != j.end()) {
          rc &= FromJsonArray(*it, _guests);
        }
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void LocalHostConfig::Clear()
    {
      _zpools.clear();
      _filesystems.clear();
      _disks.clear();
      _routes.clear();
      _guests.clear();
      _bandits.clear();
      return;
    }
    

  }  // namespace Mcrover

}  // namespace Dwm
