//===========================================================================
// @(#) $DwmPath: dwm/mcplex/mcrover/tags/mcrover-0.1.7/classes/src/DwmMcroverRPCTarget.cc 11418 $
// @(#) $Id: DwmMcroverRPCTarget.cc 11418 2020-12-12 02:54:55Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2020
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmMcroverRPCTarget.cc
//!  \author Daniel W. McRobb
//!  \brief Dwm::Mcrover::RPCTarget class implementation
//---------------------------------------------------------------------------

extern "C" {
  #include <sys/types.h>
  #include <sys/socket.h>
  #include <netinet/in.h>
  #include <arpa/inet.h>
  #include <rpc/rpc.h>
  #include <rpc/pmap_prot.h>
}

#include <mutex>
#include <thread>

#include "DwmSocket.hh"
#include "DwmSvnTag.hh"
#include "DwmSysLogger.hh"
#include "DwmMcroverRPCTarget.hh"
#include "DwmMcroverUtils.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/mcplex/mcrover/tags/mcrover-0.1.7/classes/src/DwmMcroverRPCTarget.cc 11418 $");

//---------------------------------------------------------------------------
// clnt_create_timed() doesn't exist on Linux.  The alternative
// (from TI-RPC) is a piece of crap (doesn't timeout as requested), so
// there's no point in using it.
// On FreeBSD, clnt_create_timed() is not threadsafe.
// On macos, clnt_create() takes pointer to mutable for host and protocol.
//---------------------------------------------------------------------------
#ifdef __APPLE__
  #define MY_CLIENT_CREATE(h,pn,vn,pr,to)  \
    clnt_create((char *)(h),(pn),(vn),(char *)(pr))
#else
  #define MY_CLIENT_CREATE(h,pn,vn,pr,to)  clnt_create((h),(pn),(vn),(pr))
#endif

namespace Dwm {

  namespace Mcrover {

    using namespace std;

    static mutex  g_rpcMtx;
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    RPCTarget::RPCTarget(const TargetHostConfig & hostConfig)
        : _name(hostConfig.Name()), _addr(hostConfig.Address()),
          _expectedPrograms(hostConfig.ExpectedPrograms())
    { }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const std::string & RPCTarget::Name() const
    {
      return _name;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const Dwm::Ipv4Address & RPCTarget::Address() const
    {
      return _addr;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const std::set<RPCProgramId> & RPCTarget::ExpectedPrograms() const
    {
      return _expectedPrograms;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool RPCTarget::Test(std::set<RPCProgramId> & failedPrograms) const
    {
      failedPrograms = _expectedPrograms;
      
      if (PortmapperReachable(_addr)) {
        lock_guard<mutex>  clientCreateLock(g_rpcMtx);
        struct timeval timeOut = { 2, 0 };
        string  addrStr = (string)_addr;
        CLIENT  *client = MY_CLIENT_CREATE(addrStr.c_str(), PMAPPROG,
                                           PMAPVERS, "tcp", &timeOut);
        if (client) {
          struct pmaplist *head = NULL;
          enum clnt_stat  clientStatus = 
            CLNT_CALL(client, PMAPPROC_DUMP, (xdrproc_t) xdr_void,
                      NULL, (xdrproc_t) xdr_pmaplist, (char *)&head,
                      timeOut);
          if (RPC_SUCCESS == clientStatus) {
            if (head) {
              for (auto pmptr = head; pmptr != NULL; ) {
                if ((pmptr->pml_map.pm_prot == 6)
                    || (pmptr->pml_map.pm_prot == 17)) {
                  RPCProgramId  progId(pmptr->pml_map.pm_prog,
                                       pmptr->pml_map.pm_prot,
                                       pmptr->pml_map.pm_vers);
                  auto  fpit = failedPrograms.find(progId);
                  if (fpit != failedPrograms.end()) {
                    if (Ping(progId)) {
                      failedPrograms.erase(fpit);
                    }
                  }
                }
                //  I can't figure out how to free a list of pmaplist with
                //  an RPC API call that doesn't leak, so I just free each
                //  one while I'm walking the singly-linked-list since I
                //  only traverse it once.
                auto  freeptr = pmptr;
                pmptr = pmptr->pml_next;
                free(freeptr);
              }
              //  This leaks on both Linux (Ubuntu 20.04) and FreeBSD
              //  (12.2-STABLE).  According to valgrind, it tries to free
              //  a block that's a subset of what was allocated, and not
              //  at the start of the block that was allocated.  Hence the
              //  code above that frees each pmaplist while we're looping
              //  through the singly-linked-list of pmaplist instances.
              
              // clnt_freeres(client, (xdrproc_t) xdr_pmaplist, (char *)&head);
            }
            else {
              Syslog(LOG_WARNING, "No RPC programs registered at %s",
                     addrStr.c_str());
            }
          }
          else {
            Syslog(LOG_ERR, "RPC PMAPPROC_DUMP failed for %s",
                   addrStr.c_str());
          }
          auth_destroy(client->cl_auth);
          clnt_destroy(client);
        }
        else {
          Syslog(LOG_ERR, "Failed to create RPC client for %s",
                 addrStr.c_str());
        }
      }
      else {
        Syslog(LOG_WARNING, "rpcbind unreachable at %s",
               ((string)_addr).c_str());
      }
      return (failedPrograms.empty());
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool RPCTarget::operator < (const RPCTarget & target) const
    {
      bool  rc = false;
      if (_addr < target._addr) {
        rc = true;
      }
      else if (_addr == target._addr) {
        if (_expectedPrograms < target._expectedPrograms) {
          rc = true;
        }
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool RPCTarget::operator == (const RPCTarget & target) const
    {
      return ((_addr == target._addr)
              && (_expectedPrograms == target._expectedPrograms));
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool RPCTarget::Ping(const RPCProgramId & prog) const
    {
      bool  rc = false;
      CLIENT  *client = nullptr;
      struct timeval  timeOut =  { 2, 0 };
      
      switch (prog.Protocol()) {
        case 6:
          client = MY_CLIENT_CREATE(((string)_addr).c_str(), prog.Number(),
                                    prog.Version(), "tcp", timeOut);
          break;
        case 17:
          client = MY_CLIENT_CREATE(((string)_addr).c_str(), prog.Number(),
                                    prog.Version(), "udp", &timeOut);
          break;
        default:
          break;
      }
      
      if (client) {
        timeOut.tv_sec = 2;
        timeOut.tv_usec = 0;
        enum clnt_stat  rpcStatus =
          CLNT_CALL(client, NULLPROC, (xdrproc_t) xdr_void,
                    (char *)NULL, (xdrproc_t) xdr_void,
                    (char *)NULL, timeOut);
        if (RPC_SUCCESS == rpcStatus) {
          rc = true;
        }
        auth_destroy(client->cl_auth);
        clnt_destroy(client);
      }
      else {
        Syslog(LOG_ERR, "Failed to create client for %s",
               ((string)_addr).c_str());
      }
      return rc;
    }
      
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool
    RPCTarget::PortmapperReachable(const Dwm::Ipv4Address & addr,
                                   std::chrono::milliseconds timeOut) const
    {
      using Clock = std::chrono::system_clock;
      
      bool  rc = false;
      Dwm::Socket  s;
      
      if (s.Open(PF_INET, SOCK_STREAM, 0)) {
        s.SetNonBlocking();
        auto  startTime = Clock::now();
        auto  endTime = startTime + timeOut;
        
        while (Clock::now() < endTime) {
          errno = 0;
          if (s.Connect(_addr, 111)) {
            rc = true;
            break;
          }
          else if (EISCONN == errno) {
            rc = true;
            break;
          }
          else {
            std::this_thread::sleep_for(std::chrono::milliseconds(20));
          }
        }
        s.Close();
      }
      return rc;
    }
    
  }  // namespace Mcrover

  
}  // namespace Dwm
