//===========================================================================
// @(#) $DwmPath: dwm/mcplex/mcrover/tags/mcrover-0.1.7/classes/src/DwmMcroverServiceConfig.cc 12186 $
// @(#) $Id: DwmMcroverServiceConfig.cc 12186 2023-06-21 03:04:41Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2020, 2022, 2023
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmMcroverServiceConfig.cc
//!  \author Daniel W. McRobb
//!  \brief Dwm::Mcrover::ServiceConfig class implementation
//---------------------------------------------------------------------------

#include "DwmSvnTag.hh"
#include "DwmSysLogger.hh"
#include "DwmMcroverServiceConfig.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/mcplex/mcrover/tags/mcrover-0.1.7/classes/src/DwmMcroverServiceConfig.cc 12186 $");

namespace Dwm {

  namespace Mcrover {

    using namespace std;
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const set<boost::asio::ip::tcp::endpoint> &
    ServiceConfig::Addresses() const
    {
      return _serviceAddresses;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const set<boost::asio::ip::tcp::endpoint> &
    ServiceConfig::Addresses(const set<boost::asio::ip::tcp::endpoint> & addrs)
    {
      _serviceAddresses = addrs;
      return _serviceAddresses;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void ServiceConfig::AddAddress(const boost::asio::ip::tcp::endpoint & addr)
    {
      _serviceAddresses.insert(addr);
      return;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const string & ServiceConfig::KeyDirectory() const
    {
      return _keyDirectory;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const string & ServiceConfig::KeyDirectory(const string & keyDir)
    {
      _keyDirectory = keyDir;
      return _keyDirectory;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const set<IpPrefix> & ServiceConfig::AllowedClients() const
    {
      return _allowedClients;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    set<IpPrefix> & ServiceConfig::AllowedClients()
    {
      return _allowedClients;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void ServiceConfig::Clear()
    {
      _serviceAddresses.clear();
      _keyDirectory = "/usr/local/etc/mcroverd";
      _allowedClients.clear();
      return;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    std::ostream & operator << (std::ostream & os, const ServiceConfig & cfg)
    {
      if (os) {
        os << "#============================================================================\n"
           << "#  Network service to allow authorized clients to fetch alerts.\n"
           << "#============================================================================\n"
           << "service {\n";
        if (! cfg._serviceAddresses.empty()) {
          os << "    addresses = [\n";
          auto  iter = cfg._serviceAddresses.begin();
          os << "        { address = \"" << iter->address()
             << "\"; port = " << iter->port() << "; }";
          ++iter;
          for ( ; iter != cfg._serviceAddresses.end(); ++iter) {
            os << ",\n        { address = \"" << iter->address()
               << "\"; port = " << iter->port() << "; }";
          }
          os << "\n    ];\n\n";
        }
        if (! cfg._keyDirectory.empty()) {
          os << "    keyDirectory = \"" << cfg._keyDirectory << "\";\n\n";
        }
        
        if (! cfg._allowedClients.empty()) {
          os << "    allowedClients = [\n";
          auto  iter = cfg._allowedClients.begin();
          os << "        \"" << *iter << "\"";
          ++iter;
          for ( ; iter != cfg._allowedClients.end(); ++iter) {
            os << ",\n        \"" << *iter << "\"";
          }
          os << "\n    ];\n";
        }
        os << "};\n";
      }
      return os;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    nlohmann::json ServiceConfig::ToJson() const
    {
      nlohmann::json  j;
      size_t  i = 0;
      for (auto sa : _serviceAddresses) {
        nlohmann::json  obj;
        obj["addr"] = sa.address().to_string();
        obj["port"] = sa.port();
        j["serviceAddresses"][i] = obj;
        ++i;
      }
      j["keyDirectory"] = _keyDirectory;
      i = 0;
      for (auto ac : _allowedClients) {
        j["allowedClients"][i] = (string)ac;
        ++i;
      }
      return j;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool ServiceConfig::FromJson(const nlohmann::json & j)
    {
      namespace asip = boost::asio::ip;
      
      bool  rc = false;
      if (j.is_object()) {
        auto  it = j.find("serviceAddresses");
        if ((it != j.end()) && it->is_array()) {
          size_t  i = 0;
          for ( ; i < it->size(); ++i) {
            if ((*it)[i].is_object()) {
              auto  ait = (*it)[i].find("addr");
              if (ait != (*it)[i].end()) {
                if (ait->is_string()) {
                  auto  pit = (*it)[i].find("port");
                  if ((pit != (*it)[i].end()) && pit->is_number()) {
                    uint32_t  port = pit->get<uint32_t>();
                    if (port <= 0xFFFF) {
                      asip::tcp::endpoint  ep;
                      ep.address(asip::make_address(ait->get<string>()));
                      ep.port((uint16_t)port);
                      _serviceAddresses.insert(ep);
                    }
                    else {
                      Syslog(LOG_ERR, "Bad port '%u'", port);
                      break;
                    }
                  }
                }
                else {
                  Syslog(LOG_ERR, "Bad address");
                }
              }
            }
          }
        }
        it = j.find("keyDirectory");
        if ((it != j.end()) && it->is_string()) {
          _keyDirectory = it->get<string>();
          rc = true;
        }
        it = j.find("allowedClients");
        if ((it != j.end()) && it->is_array()) {
          size_t  i = 0;
          for ( ; i < it->size(); ++i) {
            if ((*it)[i].is_string()) {
              IpPrefix  pfx((*it)[i].get<string>());
              _allowedClients.insert(pfx);
            }
            else {
              break;
            }
          }
        }
      }
      return rc;
    }

  }  // namespace Mcrover

}  // namespace Dwm
